// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_utility.h
 *  \brief      MGL ファイルユーティリティ
 *  \date       Since: September 29, 2022. 14:53:57 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FILE_UTILITY_H_1664430837
#define INCGUARD_MGL_FILE_UTILITY_H_1664430837

#include <mgl/file/mgl_file_mounter.h>
#include <mgl/stl/mgl_stl_string.h>

namespace MGL::File
{
//! ファイルユーティリティクラス
class Utility
{
public:
    [[nodiscard]] size_t GetSize(const PathView &path) noexcept;
    const Result &MakeDirectory(const PathView &path) noexcept;
    const Result &Move(const PathView &sourcePath, const PathView &destPath) noexcept;
    const Result &Remove(const PathView &path) noexcept;
    const Result &Copy(const PathView &sourcePath, const PathView &destPath) noexcept;
    [[nodiscard]] bool Exists(const PathView &path) noexcept;
    [[nodiscard]] bool IsSystemNativeFile(const PathView &path) noexcept;
    const Result &Mount(const PathView &mountName, const PathView &path, MountAccessType accessType, DelegateKey delegateKey = Mounter::kDefaultDelegateKey) noexcept;
    const Result &Unmount(const PathView &mountName) noexcept;
    const Result &Remount(const PathView &mountName, const PathView &path, MountAccessType accessType, DelegateKey delegateKey = Mounter::kDefaultDelegateKey) noexcept;
    [[nodiscard]] static bool IsMounted(const PathView &mountName) noexcept;
    [[nodiscard]] STL::string GetSystemNativePath(const PathView &path) noexcept;
    const Result &RemoveDelegate(DelegateKey key) noexcept;
    static void SetDefaultDelegate(DelegateKey key) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デリゲートの追加
     *  \tparam     DelegateClass       追加するデリゲートクラス
     *  \param[in]  key                 デリゲートキー
     *  \param[in]  args                デリゲートクラスのコンストラクタに渡す引数
     *  \return     失敗時にはいずれかのエラーが設定される
     */
    /* ------------------------------------------------------------------------- */
    template <class DelegateClass, class... Args>
    constexpr const Result &AddDelegate(DelegateKey key, Args... args) noexcept
    {
        _result = Mounter::GetInstance().AddDelegate(key, STL::make_shared<DelegateClass>(args...));
        return _result;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      処理結果の取得
     *  \return     最後に実行した処理の処理結果
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Result &GetResult() const noexcept
    {
        return _result;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      最後の処理でエラーが発生しているかの取得
     *  \retval     true    エラーが発生している
     *  \retval     false   エラーは発生していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool HasError() const noexcept
    {
        return _result.HasError();
    }

private:
    Result _result;
};
}    // namespace MGL::File

#endif    // INCGUARD_MGL_FILE_UTILITY_H_1664430837

// vim: et ts=4 sw=4 sts=4
