// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_throwing_utility.h
 *  \brief      MGL 例外を発生させるファイルユーティリティ
 *  \date       Since: February 5, 2021. 16:22:08 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FILE_THROWING_UTILITY_H_1612509728
#define INCGUARD_MGL_FILE_THROWING_UTILITY_H_1612509728

#include <mgl/file/mgl_file_exception.h>
#include <mgl/file/mgl_file_mounter.h>
#include <mgl/text/mgl_text_format.h>

namespace MGL::File
{
//! 例外を発生させるファイルユーティリティクラス
class ThrowingUtility
{
public:
    [[nodiscard]] size_t GetSize(const PathView &path);
    void MakeDirectory(const PathView &path);
    void Move(const PathView &sourcePath, const PathView &destPath);
    void Remove(const PathView &path);
    void Copy(const PathView &sourcePath, const PathView &destPath);
    [[nodiscard]] bool Exists(const PathView &path);
    [[nodiscard]] bool IsSystemNativeFile(const PathView &path);
    void Mount(const PathView &mountName, const PathView &path, MountAccessType accessType, DelegateKey delegateKey = Mounter::kDefaultDelegateKey);
    void Unmount(const PathView &mountName);
    void Remount(const PathView &mountName, const PathView &path, MountAccessType accessType, DelegateKey delegateKey = Mounter::kDefaultDelegateKey);
    [[nodiscard]] static bool IsMounted(const PathView &mountName) noexcept;
    [[nodiscard]] STL::string GetSystemNativePath(const PathView &path);
    void RemoveDelegate(DelegateKey key);
    static void SetDefaultDelegate(DelegateKey key) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デリゲートの追加
     *  \tparam     DelegateClass       追加するデリゲートクラス
     *  \param[in]  key                 デリゲートキー
     *  \param[in]  args                デリゲートクラスのコンストラクタに渡す引数
     *  \exception  MGL::File::Result   失敗時にエラーが設定されたリザルトを返す
     */
    /* ------------------------------------------------------------------------- */
    template <class DelegateClass, class... Args>
    constexpr void AddDelegate(DelegateKey key, Args... args)
    {
        _result = Mounter::GetInstance().AddDelegate(key, STL::make_shared<DelegateClass>(args...));
        if (_result.HasError())
        {
            throw Exception(_result.GetError(), Text::Format("Failed to add delegate: {}", static_cast<uint32_t>(key)).c_str());
        }
    }

private:
    Result _result;
};
}    // namespace MGL::File
#endif    // INCGUARD_MGL_FILE_THROWING_UTILITY_H_1612509728

// vim: et ts=4 sw=4 sts=4
