// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_throwing_handle.h
 *  \brief      MGL 例外を発生させるファイルハンドル
 *  \date       Since: February 5, 2021. 18:14:23 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FILE_THROWING_HANDLE_H_1612516463
#define INCGUARD_MGL_FILE_THROWING_HANDLE_H_1612516463

#include <mgl/file/mgl_file_mounter.h>

namespace MGL::File
{
//! 例外を発生させるファイルハンドルクラス
class ThrowingHandle
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr ThrowingHandle() noexcept
        : _work(nullptr)
    {
    }

    ThrowingHandle(const PathView &path, OpenMode mode = OpenMode::Read);
    void Open(const PathView &path, OpenMode mode = OpenMode::Read);
    void Close();

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ファイルがオープンされているかを取得
     *  \retval     true    オープンされている
     *  \retval     false   オープンされていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] MGL_MAYBE_CONSTEXPR bool IsOpen() const noexcept
    {
        return (_work) ? _work->IsOpen() : false;
    }

    size_t Read(void *buffer, size_t size);
    size_t Write(const void *buffer, size_t size);
    size_t Seek(SeekType seekType, int32_t offset);
    size_t Seek(SeekType seekType, size_t offset);
    size_t Skip(size_t size);
    [[nodiscard]] size_t GetOffset();
    [[nodiscard]] bool IsEOF();
    [[nodiscard]] size_t GetSize();

    // コピー禁止
    ThrowingHandle(const ThrowingHandle &) = delete;
    ThrowingHandle &operator=(const ThrowingHandle &) = delete;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ムーブコンストラクタ
     *  \param[in]  other   移動元ハンドル
     */
    /* ------------------------------------------------------------------------- */
    ThrowingHandle(ThrowingHandle &&other) noexcept
        : _work(std::move(other._work))
    {
        other._work.reset();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ムーブ代入演算
     *  \param[in]  other   移動元ハンドル
     */
    /* ------------------------------------------------------------------------- */
    ThrowingHandle &operator=(ThrowingHandle &&other) noexcept
    {
        if (this != &other)
        {
            Close();

            _work = std::move(other._work);

            other._work.reset();
        }

        return *this;
    }

private:
    AccessWorkPtr _work;
    Result _result;
};
}    // namespace MGL::File
#endif    // INCGUARD_MGL_FILE_THROWING_HANDLE_H_1612516463

// vim: et ts=4 sw=4 sts=4
