// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_path_view.h
 *  \brief      文字列の参照のみを行うファイルパスクラス
 *  \date       Since: July 30, 2023. 19:50:17 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FILE_PATH_VIEW_H_1690714217
#define INCGUARD_MGL_FILE_PATH_VIEW_H_1690714217

#include <mgl/file/mgl_file_path.h>

namespace MGL::File
{
//! 文字列の参照のみを行うファイルパスクラス
class PathView
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  path    参照するパスの文字列
     */
    /* ------------------------------------------------------------------------- */
    constexpr PathView(const char *path) noexcept
        : _path(path)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  path    参照するパスの文字列
     */
    /* ------------------------------------------------------------------------- */
    PathView(const Path &path) noexcept
        : _path(path.GetCString())
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      const char *型へのキャスト
     *  \return     const char *型の文字列
     */
    /* ------------------------------------------------------------------------- */
    constexpr operator const char *() const noexcept
    {
        return _path;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      C言語の文字列として取得
     *  \return     const char *型の文字列
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const char *GetCString() const noexcept
    {
        return _path;
    }

    [[nodiscard]] Path Append(const char *element) const noexcept;
    [[nodiscard]] Path Concat(const char *string) const noexcept;
    [[nodiscard]] Path Concat(char character) const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      要素の追加
     *  \param[in]  element     追加する要素
     *  \return     追加後のパス
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] Path operator /=(const char *element) const noexcept
    {
        return Append(element);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      文字列の追加
     *  \param[in]  string  追加する文字列
     *  \return     追加後のパス
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] Path operator +=(const char *string) const noexcept
    {
        return Concat(string);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      文字の追加
     *  \param[in]  character   追加する文字
     *  \return     追加後のパス
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] Path operator +=(char character) const noexcept
    {
        return Concat(character);
    }

    [[nodiscard]] size_t GetLength() const noexcept;
    [[nodiscard]] bool IsEmpty() const noexcept;

    [[nodiscard]] Path GetRoot() const noexcept;
    [[nodiscard]] Path GetMountName() const noexcept;
    [[nodiscard]] Path GetRelativePath() const noexcept;
    [[nodiscard]] Path GetParent() const noexcept;
    [[nodiscard]] Path GetFilename() const noexcept;
    [[nodiscard]] Path GetStem() const noexcept;
    [[nodiscard]] Path GetExtension() const noexcept;

private:
    const char *_path;
};

}   // namespace MGL::File

#endif	// INCGUARD_MGL_FILE_PATH_VIEW_H_1690714217

// vim: et ts=4 sw=4 sts=4
