// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_path.h
 *  \brief      MGL ファイルパスクラス
 *  \date       Since: July 26, 2023. 15:54:39 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FILE_PATH_H_1690354479
#define INCGUARD_MGL_FILE_PATH_H_1690354479

#include <mgl/stl/mgl_stl_string.h>

namespace MGL::File
{
//! ファイルパスクラス
class Path
{
public:
    Path(size_t reserveSize = 0) noexcept;
    Path(const char *path) noexcept;
    Path(const char *path, size_t reserveSize) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      const char *型へのキャスト
     *  \return     const char *型の文字列
     */
    /* ------------------------------------------------------------------------- */
    operator const char *() const noexcept
    {
        return _path.c_str();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      C言語の文字列として取得
     *  \return     const char *型の文字列
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const char *GetCString() const noexcept
    {
        return _path.c_str();
    }

    Path &Append(const char *element) noexcept;
    Path &Concat(const char *string) noexcept;
    Path &Concat(char character) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      要素の追加
     *  \param[in]  element     追加する要素
     *  \return     追加後のパス
     */
    /* ------------------------------------------------------------------------- */
    Path &operator/=(const char *element) noexcept
    {
        return Append(element);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      文字列の追加
     *  \param[in]  string  追加する文字列
     *  \return     追加後のパス
     */
    /* ------------------------------------------------------------------------- */
    Path &operator+=(const char *string) noexcept
    {
        return Concat(string);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      文字の追加
     *  \param[in]  character   追加する文字
     *  \return     追加後のパス
     */
    /* ------------------------------------------------------------------------- */
    Path &operator+=(char character) noexcept
    {
        return Concat(character);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      パスの長さを取得
     *  \return     パスの長さ（終端記号を除くchar型の要素数）
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] size_t GetLength() const noexcept
    {
        return _path.length();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      パスが空であるかを取得
     *  \retval     true    空である
     *  \retval     false   空ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsEmpty() const noexcept
    {
        return _path.empty();
    }

    [[nodiscard]] Path GetRoot() const noexcept;
    [[nodiscard]] Path GetMountName() const noexcept;
    [[nodiscard]] Path GetRelativePath() const noexcept;
    [[nodiscard]] Path GetParent() const noexcept;
    [[nodiscard]] Path GetFilename() const noexcept;
    [[nodiscard]] Path GetStem() const noexcept;
    [[nodiscard]] Path GetExtension() const noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      要素を削除
     */
    /* ------------------------------------------------------------------------- */
    void Clear() noexcept
    {
        _path.clear();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      パスを設定
     *  \param[in]  path    設定するパス
     */
    /* ------------------------------------------------------------------------- */
    void Set(const char *path) noexcept
    {
        _path = path;
    }

private:
    MGL::STL::string _path;
};

}    // namespace MGL::File

#endif    // INCGUARD_MGL_FILE_PATH_H_1690354479

// vim: et ts=4 sw=4 sts=4
