// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_mounter.h
 *  \brief      MGL ファイルマウンタ
 *  \date       Since: January 20, 2021. 10:14:25 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FILE_MOUNTER_H_1611105265
#define INCGUARD_MGL_FILE_MOUNTER_H_1611105265

#include <mutex>

#include <mgl/common/mgl_singleton.h>
#include <mgl/file/mgl_file_delegate.h>
#include <mgl/file/mgl_file_path_view.h>
#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/stl/mgl_stl_hash.h>
#include <mgl/stl/mgl_stl_memory.h>
#include <mgl/stl/mgl_stl_string.h>

namespace MGL::File
{
//! ファイルマウンタ
class Mounter final : public SharedSingleton<Mounter>
{
public:
    //! マウント情報
    struct MountProperty
    {
        Path name;                                                //!< マウント名
        Path path;                                                //!< マウントパス
        MountAccessType accessType{MountAccessType::ReadOnly};    //!< アクセスタイプ
        SharedDelegate delegate{nullptr};                         //!< ファイルデリゲート
        SharedMountWork mountWork{nullptr};                       //!< マウント毎のワーク
    };

    //! マウント情報の共有ポインタ
    using SharedMountProperty = std::shared_ptr<MountProperty>;

    //! デフォルトを表す予約デリゲートキー
    static constexpr DelegateKey kDefaultDelegateKey = MakeDelegateKey("MGL-Default");

    [[nodiscard]] static STL::unique_ptr<Mounter> &GetInstanceRef() noexcept;

    Result Mount(const PathView &mountName, const PathView &path, MountAccessType accessType, DelegateKey delegateKey = kDefaultDelegateKey) noexcept;
    Result Unmount(const PathView &mountName) noexcept;
    [[nodiscard]] SharedMountProperty Get(const PathView &mountName) noexcept;
    [[nodiscard]] bool IsMounted(const PathView &mountName) noexcept;

    Result AddDelegate(DelegateKey key, const SharedDelegate &delegate) noexcept;
    Result RemoveDelegate(DelegateKey key) noexcept;
    [[nodiscard]] SharedDelegate GetDelegate(DelegateKey key = kDefaultDelegateKey) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デフォルトのデリゲートを設定
     *  \param[in]  key     デフォルトのデリゲートキー
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetDefaultDelegate(DelegateKey key) noexcept
    {
        _defaultDelegateKey = key;
    }

private:
    STL::unordered_map<STL::string, std::shared_ptr<MountProperty>, STL::Hash::string> _mountProperties;
    std::recursive_mutex _mountMutex;

    STL::unordered_map<DelegateKey, SharedDelegate> _delegates;
    std::mutex _delegateMutex;

    DelegateKey _defaultDelegateKey{kDefaultDelegateKey};
};
}    // namespace MGL::File
#endif    // INCGUARD_MGL_FILE_MOUNTER_H_1611105265

// vim: et ts=4 sw=4 sts=4
