// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_handle.h
 *  \brief      MGL ファイルハンドル
 *  \date       Since: January 23, 2021. 10:17:52 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FILE_HANDLE_H_1611364672
#define INCGUARD_MGL_FILE_HANDLE_H_1611364672

#include <mgl/file/mgl_file_mounter.h>

namespace MGL::File
{
//! ファイルハンドルクラス
class Handle
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr Handle() noexcept
        : _work(nullptr)
    {
    }

    Handle(const PathView &path, OpenMode mode = OpenMode::Read) noexcept;
    Result Open(const PathView &path, OpenMode mode = OpenMode::Read) noexcept;
    Result Close() noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ファイルがオープンされているかを取得
     *  \retval     true    オープンされている
     *  \retval     false   オープンされていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] MGL_MAYBE_CONSTEXPR bool IsOpen() const noexcept
    {
        return (_work) ? _work->IsOpen() : false;
    }

    size_t Read(void *buffer, size_t size) noexcept;
    size_t Write(const void *buffer, size_t size) noexcept;
    size_t Seek(SeekType seekType, int32_t offset) noexcept;
    size_t Seek(SeekType seekType, size_t offset) noexcept;
    size_t Skip(size_t size) noexcept;
    [[nodiscard]] size_t GetOffset() noexcept;
    [[nodiscard]] bool IsEOF() noexcept;
    [[nodiscard]] size_t GetSize() noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      処理結果の取得
     *  \return     最後に実行した処理の処理結果
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Result &GetResult() const noexcept
    {
        return _result;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      最後の処理でエラーが発生しているかの取得
     *  \retval     true    エラーが発生している
     *  \retval     false   エラーは発生していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool HasError() const noexcept
    {
        return _result.HasError();
    }

    // コピー禁止
    Handle(const Handle &) = delete;
    Handle &operator=(const Handle &) = delete;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ムーブコンストラクタ
     *  \param[in]  other   移動元ハンドル
     */
    /* ------------------------------------------------------------------------- */
    Handle(Handle &&other) noexcept
        : _work(std::move(other._work))
        , _result(other._result)
    {
        other._work.reset();
        other._result = Result();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ムーブ代入演算
     *  \param[in]  other   移動元ハンドル
     */
    /* ------------------------------------------------------------------------- */
    Handle &operator=(Handle &&other) noexcept
    {
        if (this != &other)
        {
            Close();

            _work = std::move(other._work);
            _result = other._result;

            other._work.reset();
            other._result = Result();
        }

        return *this;
    }

private:
    AccessWorkPtr _work;
    Result _result;
};

}    // namespace MGL::File

#endif    // INCGUARD_MGL_FILE_HANDLE_H_1611364672

// vim: et ts=4 sw=4 sts=4
