// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_delegate.h
 *  \brief      MGL ファイルデリゲート
 *  \date       Since: January 20, 2021. 9:56:02 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FILE_DELEGATE_H_1611104162
#define INCGUARD_MGL_FILE_DELEGATE_H_1611104162

#include <cstddef>

#include <mgl/file/mgl_file_access_work.h>
#include <mgl/file/mgl_file_defs.h>
#include <mgl/file/mgl_file_mount_work.h>
#include <mgl/file/mgl_file_path_view.h>

namespace MGL::File
{
//! ファイルデリゲートクラス
class Delegate
{
public:
    virtual ~Delegate() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このデリゲートがシステム標準のファイルを扱うものかを取得
     *  \retval     true    システム標準のファイルを扱う
     *  \retval     false   システム標準ではないファイルを扱う
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsManagedSystemNativeFile() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このデリゲートが書き込みに対応しているかを取得
     *  \retval     true    対応している
     *  \retval     false   対応していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsWritable() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウント時の処理
     *  \param[out] mountWork   マウント毎に使用するワークの共有ポインタ
     *  \param[in]  path        マウントパス
     *  \param[in]  accessType  アクセスタイプ
     *  \return     エラー発生時にいずれかのエラーが設定される
     */
    /* ------------------------------------------------------------------------- */
    virtual Result Mount(SharedMountWork &mountWork, const PathView &path, MountAccessType accessType) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウント解除時の処理
     *  \param[out] mountWork   マウント毎に使用するワークの共有ポインタ
     */
    /* ------------------------------------------------------------------------- */
    virtual void Unmount(SharedMountWork &mountWork) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          ファイルをオープン
     *  \param[out]     result      処理結果
     *  \param[in,out]  mountWork   マウント毎のワーク
     *  \param[in]      path        オープンするファイルのパス
     *  \param[in]      mode        オープンモード
     *  \return         オープンしたファイルのワーク
     */
    /* ------------------------------------------------------------------------- */
    virtual AccessWorkPtr Open(Result &result, SharedMountWork &mountWork, const PathView &path, OpenMode mode) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          ファイルをクローズ
     *  \param[in,out]  work    クローズするファイルのワーク
     *  \return         エラー発生時にいずれかのエラーが設定される
     */
    /* ------------------------------------------------------------------------- */
    virtual Result Close(AccessWork *work) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          ファイルを読み込み
     *  \param[in,out]  work        アクセスするファイルのワーク
     *  \param[out]     result      処理結果
     *  \param[in]      buffer      読み込み先のアドレス
     *  \param[in]      size        読み込みサイズ
     *  \return         実際に読み込んだバイト数
     */
    /* ------------------------------------------------------------------------- */
    virtual size_t Read(AccessWork *work, Result &result, void *buffer, size_t size) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          ファイルを書き込み
     *  \param[in,out]  work        アクセスするファイルのワーク
     *  \param[out]     result      処理結果
     *  \param[out]     buffer      書き込み先のアドレス
     *  \param[in]      size        書き込みサイズ
     *  \return         実際に書き込んだバイト数
     */
    /* ------------------------------------------------------------------------- */
    virtual size_t Write(AccessWork *work, Result &result, const void *buffer, size_t size) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          ストリーム位置を設定
     *  \param[in,out]  work        アクセスするファイルのワーク
     *  \param[out]     result      処理結果
     *  \param[in]      seekType    シークタイプ
     *  \param[in]      offset      オフセット
     *  \return         エラー発生時にはいずれかのエラーが設定される
     */
    /* ------------------------------------------------------------------------- */
    virtual size_t Seek(AccessWork *work, Result &result, SeekType seekType, int32_t offset) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ストリーム位置を取得
     *  \param[in]  work    アクセスするファイルのワーク
     *  \param[out] result  エラー発生時の結果の格納先
     *  \return     現在のストリーム位置
     */
    /* ------------------------------------------------------------------------- */
    virtual size_t GetOffset(AccessWork *work, Result &result) const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          ファイルストリームが終端に達しているかを取得
     *  \param[in,out]  work    アクセスするファイルのワーク
     *  \param[out]     result  エラー発生時の結果の格納先
     *  \retval         true    終端に達している
     *  \retval         false   終端に達していない
     */
    /* ------------------------------------------------------------------------- */
    virtual bool IsEOF(AccessWork *work, Result &result) const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      オープンしているファイルのサイズを取得
     *  \param[in]  work    アクセスするファイルのワーク
     *  \param[out] result  エラー発生時の結果の格納先
     *  \return     ファイルサイズ
     */
    /* ------------------------------------------------------------------------- */
    virtual size_t GetSize(AccessWork *work, Result &result) const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          ファイルのサイズを取得
     *  \param[in,out]  mountWork   マウント毎のワーク
     *  \param[out]     result      エラー発生時の結果の格納先
     *  \param[in]      path        サイズを取得するファイルのパス
     *  \return         ファイルサイズ
     */
    /* ------------------------------------------------------------------------- */
    virtual size_t GetSize(MountWork *mountWork, Result &result, const PathView &path) const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          ディレクトリを作成する
     *  \param[in,out]  mountWork   マウント毎のワーク
     *  \param[in]      path        作成するディレクトリのパス
     *  \return         エラー発生時にはいずれかのエラーが設定される
     */
    /* ------------------------------------------------------------------------- */
    virtual Result MakeDirectory(MountWork *mountWork, const PathView &path) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          ファイルの移動・リネーム
     *  \param[in,out]  mountWork   マウント毎のワーク
     *  \param[in]      sourcePath  移動元のパス
     *  \param[in]      destPath    移動先のパス
     *  \return         エラー発生時にはいずれかのエラーが設定される
     */
    /* ------------------------------------------------------------------------- */
    virtual Result Move(MountWork *mountWork, const PathView &sourcePath, const PathView &destPath) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          ファイルの削除
     *  \param[in,out]  mountWork   マウント毎のワーク
     *  \param[in]      path        削除するファイルのパス
     *  \return         エラー発生時にはいずれかのエラーが設定される
     */
    /* ------------------------------------------------------------------------- */
    virtual Result Remove(MountWork *mountWork, const PathView &path) noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief          ファイルの存在をチェック
     *  \param[in,out]  mountWork   マウント毎のワーク
     *  \param[out]     result      エラー発生時の結果の格納先
     *  \param[in]      path        チェックするファイルのパス
     *  \return         存在する場合はエラーなしのリザルトが返る
     */
    /* ------------------------------------------------------------------------- */
    virtual bool Exists(MountWork *mountWork, Result &result, const PathView &path) noexcept = 0;
};

}    // namespace MGL::File

#endif    // INCGUARD_MGL_FILE_DELEGATE_H_1611104162

// vim: et ts=4 sw=4 sts=4
