// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_defs.h
 *  \brief      MGL ファイル用各種定義
 *  \date       Since: January 20, 2021. 10:05:04 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FILE_DEFS_H_1611104704
#define INCGUARD_MGL_FILE_DEFS_H_1611104704

#include <cstdint>
#include <memory>

#include <mgl/common/mgl_result_wrapper.h>
#include <mgl/hash/mgl_hash_fnv1a.h>

namespace MGL::File
{
//! デリゲートキーの型
enum class DelegateKey : uint32_t {};

//! デリゲートキーのハッシュ生成に使用するシード値
constexpr uint32_t kDefaultDelegateKeySeed = MGL::Hash::kFNV1aDefaultValue32;

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デリゲートキーの生成
 *  \param[in]  keyString   キーの文字列
 *  \param[in]  seed        シード値（省略時はkDefaultDelegateKeySeed）
 *  \return     生成したデリゲートキー
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] constexpr DelegateKey MakeDelegateKey(const char *keyString, uint32_t seed = kDefaultDelegateKeySeed) noexcept
{
    return DelegateKey{MGL::Hash::FNV1a(keyString, seed)};
}

class Delegate;

//! デリゲートの共有ポインタ
using SharedDelegate = std::shared_ptr<Delegate>;

//! オープンモード
enum class OpenMode : uint8_t
{
    None,    //!< 無効
    Read,    //!< 読み込み
    Write    //!< 書き込み
};

//! マウント時のアクセスタイプ
enum class MountAccessType : uint8_t
{
    ReadOnly,    //!< 読み込み専用
    Writable,    //!< 書き込み可能
};

//! シークタイプ
enum class SeekType : uint8_t
{
    Top,       //!< 先頭から
    End,       //!< 末尾から
    Current    //!< 現在の位置から
};

//! エラー定義
enum class Error : uint16_t
{
    None                    = 0,    //!< エラーなし（成功）

    NoOperation             = 100,  //!< 処理していない
    NotImplemented          = 101,  //!< 未実装
    NotSupported            = 102,  //!< 指定された操作に対応していない
    InvalidDelegate         = 103,  //!< デリゲートが不正
    InvalidArgument         = 104,  //!< 引数が不正
    InvalidPathFormat       = 105,  //!< パスの書式が不正
    BadImplement            = 106,  //!< 実装上の不正

    DelegateAlreadyExist    = 200,  //!< デリゲートが既に存在している
    DelegateNotExist        = 201,  //!< デリゲートが存在しない
    NotMounted              = 202,  //!< マウントされていない
    AlreadyMounted          = 203,  //!< 既にマウントされている
    MountNotWritable        = 204,  //!< 書き込み不可能なマウントに対する書き込み操作を行おうとしている
    MountFailed             = 205,  //!< マウントに失敗
    NotOpened               = 206,  //!< オープンされていない
    NotReadableMode         = 207,  //!< 読み込み用にオープンされていない
    NotWritableMode         = 208,  //!< 書き込み用にオープンされていない
    NoSystemNativeFile      = 209,  //!< 対象がシステム標準のファイルではない
    NoRegularFile           = 210,  //!< 通常ファイルではない

    FileNotExist            = 300,  //!< ファイルが存在しない
    FileAlreadyExist        = 301,  //!< ファイルが既に存在している
    PathNotFound            = 302,  //!< パスが存在しない
    NotPermitted            = 303,  //!< ファイルにアクセスする権限がない
    NoSpaceStorage          = 304,  //!< 保存領域の容量不足
    ReadError               = 305,  //!< 読み込みに失敗
    WriteError              = 306,  //!< 書き込みに失敗

    UnknownError            = 999,  //!< 不明なエラー
};

//! ファイル関連の処理結果
using Result = MGL::ResultWrapper<Error, Error::None, Error::NoOperation>;

}   // namespace MGL::File
#endif	// INCGUARD_MGL_FILE_DEFS_H_1611104704

// vim: et ts=4 sw=4 sts=4
