// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_accessor.h
 *  \brief      MGL ファイルアクセサ
 *  \date       Since: January 20, 2021. 11:21:53 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FILE_ACCESSOR_H_1611109313
#define INCGUARD_MGL_FILE_ACCESSOR_H_1611109313

#include <mgl/file/mgl_file_access_work.h>
#include <mgl/file/mgl_file_mounter.h>
#include <mgl/file/mgl_file_path_view.h>
#include <mgl/mgl_environment.h>
#include <mgl/stl/mgl_stl_containers.h>

namespace MGL::File::Accessor
{
AccessWorkPtr Open(Result &result, const PathView &path, OpenMode mode) noexcept;
Result Close(AccessWorkPtr &work) noexcept;

size_t Read(AccessWorkPtr &work, Result &result, void *buffer, size_t size) noexcept;
size_t Write(AccessWorkPtr &work, Result &result, const void *buffer, size_t size) noexcept;
size_t Seek(AccessWorkPtr &work, Result &result, SeekType seekType, int32_t offset) noexcept;
size_t GetOffset(AccessWorkPtr &work, Result &result) noexcept;
bool IsEOF(AccessWorkPtr &work, Result &result) noexcept;
size_t GetSize(AccessWorkPtr &work, Result &result) noexcept;

size_t GetSize(Result &result, const PathView &path) noexcept;
Result MakeDirectory(const PathView &path) noexcept;
Result Move(const PathView &sourcePath, const PathView &destPath) noexcept;
Result Remove(const PathView &path) noexcept;
Result Copy(const PathView &sourcePath, const PathView &destPath) noexcept;
bool Exists(Result &result, const PathView &path) noexcept;
bool IsSystemNativeFile(Result &result, const PathView &path) noexcept;

Result Mount(const PathView &mountName, const PathView &path, MountAccessType accessType, DelegateKey delegateKey = Mounter::kDefaultDelegateKey) noexcept;
Result Unmount(const PathView &mountName) noexcept;
Result Remount(const PathView &mountName, const PathView &path, MountAccessType accessType, DelegateKey delegateKey = Mounter::kDefaultDelegateKey) noexcept;

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      マウントされているかを取得
 *  \param[in]  mountName   マウント名
 *  \retval     true        マウントされている
 *  \retval     false       マウントされていない
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline bool IsMounted(const PathView &mountName) noexcept
{
    return Mounter::GetInstance().IsMounted(mountName);
}

STL::string GetSystemNativePath(Result &result, const PathView &path) noexcept;

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デリゲートの追加
 *  \tparam     DelegateClass   追加するデリゲートクラス
 *  \param[in]  key             デリゲートキー
 *  \param[in]  args            デリゲートクラスのコンストラクタに渡す引数
 *  \return     失敗時にはいずれかのエラーが設定される
 */
/* ------------------------------------------------------------------------- */
template<class DelegateClass, class... Args>
constexpr Result AddDelegate(DelegateKey key, Args... args) noexcept
{
    return Mounter::GetInstance().AddDelegate(key, STL::make_shared<DelegateClass>(args...));
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デリゲートの削除
 *  \param[in]  key     削除するデリゲートキー
 *  \return     失敗時にはいずれかのエラーが設定される．存在しないデリゲートキーに対しては何もしない
 */
/* ------------------------------------------------------------------------- */
inline Result RemoveDelegate(DelegateKey key) noexcept
{
    return Mounter::GetInstance().RemoveDelegate(key);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      デフォルトのデリゲートを設定
 *  \param[in]  key     デフォルトのデリゲートキー
 */
/* ------------------------------------------------------------------------- */
inline void SetDefaultDelegate(DelegateKey key) noexcept
{
    Mounter::GetInstance().SetDefaultDelegate(key);
}
}    // namespace MGL::File::Accessor

#endif    // INCGUARD_MGL_FILE_ACCESSOR_H_1611109313

// vim: et ts=4 sw=4 sts=4
