// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_access_work.h
 *  \brief      MGL ファイルアクセスのための作業用クラス
 *  \date       Since: January 20, 2021. 10:02:48 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FILE_ACCESS_WORK_H_1611104568
#define INCGUARD_MGL_FILE_ACCESS_WORK_H_1611104568

#include <utility>

#include <mgl/file/mgl_file_defs.h>
#include <mgl/mgl_environment.h>
#include <mgl/stl/mgl_stl_memory.h>

namespace MGL::File
{
//! ファイルアクセスのための作業用クラス
class AccessWork
{
public:
    AccessWork(OpenMode mode, DelegateKey key) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デストラクタ
     */
    /* ------------------------------------------------------------------------- */
    virtual ~AccessWork() noexcept
    {
        Close();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      オープンモードの取得
     *  \return     オープンモード
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr OpenMode GetOpenMode() const noexcept
    {
        return _mode;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デリゲートキーの取得
     *  \return     デリゲートキー
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr DelegateKey GetDelegateKey() const noexcept
    {
        return _key;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デリゲートを設定
     *  \param[in]  delegate    デリゲート
     */
    /* ------------------------------------------------------------------------- */
    void SetDelegate(SharedDelegate delegate) noexcept
    {
        if (_delegate == nullptr)
        {
            _delegate = std::move(delegate);
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デリゲートを取得
     *  \return     デリゲート
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] Delegate *GetDelegate() const noexcept
    {
        return _delegate.get();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このワークがオープンされているかを取得
     *  \retval     true    オープンされている
     *  \retval     false   オープンされていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsOpen() const noexcept
    {
        return (_mode != OpenMode::None);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このワークが読み込み用にオープンされているかを取得
     *  \retval     true    読み込み用にオープンされている
     *  \retval     false   読み込み用にオープンされていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsReadable() const noexcept
    {
        return (_mode == OpenMode::Read);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このワークが書き込み用にオープンされているかを取得
     *  \retval     true    書き込み用にオープンされている
     *  \retval     false   書き込み用にオープンされていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsWritable() const noexcept
    {
        return (_mode == OpenMode::Write);
    }

    void Close() noexcept;

private:
    OpenMode _mode;
    DelegateKey _key;
    SharedDelegate _delegate;
};

//! ファイルアクセスのためのワークのポインタ型
using AccessWorkPtr = STL::unique_ptr<AccessWork>;

}    // namespace MGL::File
#endif    // INCGUARD_MGL_FILE_ACCESS_WORK_H_1611104568

// vim: et ts=4 sw=4 sts=4
