// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_delegate_win32.h
 *  \brief      MGL Win32用ファイルデリゲート
 *  \date       Since: March 27, 2021. 4:25:50 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FILE_DELEGATE_WIN32_H_1616786750
#define INCGUARD_MGL_FILE_DELEGATE_WIN32_H_1616786750

#include <mgl/mgl_environment.h>
#if defined(MGL_FILE_DELEGATE_ENABLE_WIN32)

#include <mgl/file/mgl_file_delegate.h>
#include <mgl/stl/mgl_stl_string.h>

namespace MGL::File
{
//! Win32用ファイルデリゲートクラス
class Win32Delegate : public Delegate
{
public:
    //! このデリゲートを表すデリゲートキー
    static constexpr DelegateKey kDelegateKey = MakeDelegateKey("MGL-Win32");

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このデリゲートがシステム標準のファイルを扱うものかを取得
     *  \retval     true    システム標準のファイルを扱う
     *  \retval     false   システム標準ではないファイルを扱う
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsManagedSystemNativeFile() const noexcept override
    {
        return true;
    }
        
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このデリゲートが書き込みに対応しているかを取得
     *  \retval     true    対応している
     *  \retval     false   対応していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsWritable() const noexcept override
    {
        return true;    // 対応している
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウント時の処理
     *  \param[out] mountWork   マウント毎に使用するワークの共有ポインタ
     *  \param[in]  path        マウントパス
     *  \param[in]  accessType  アクセスタイプ
     *  \return     エラー発生時にいずれかのエラーが設定される
     */
    /* ------------------------------------------------------------------------- */
    Result Mount([[maybe_unused]] SharedMountWork &mountWork, [[maybe_unused]] const PathView &path, [[maybe_unused]] MountAccessType accessType) noexcept override
    {
        return Result::Succeeded();     // このデリゲートはマウントワークを使用しないので何もせずに成功を返す
    }
    
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウント解除時の処理
     *  \param[out] mountWork   マウント毎に使用するワークの共有ポインタ
     */
    /* ------------------------------------------------------------------------- */
    void Unmount([[maybe_unused]] SharedMountWork &mountWork) noexcept override
    {
    }

    AccessWorkPtr Open(Result &result, SharedMountWork &mountWork, const PathView &path, OpenMode mode) noexcept override;
    Result Close(AccessWork *work) noexcept override;

    size_t Read(AccessWork *work, Result &result, void *buffer, size_t size) noexcept override;
    size_t Write(AccessWork *work, Result &result, const void *buffer, size_t size) noexcept override;
    size_t Seek(AccessWork *work, Result &result, SeekType seekType, int32_t offset) noexcept override;
    size_t GetOffset(AccessWork *work, Result &result) const noexcept override;
    bool IsEOF(AccessWork *work, Result &result) const noexcept override;
    size_t GetSize(AccessWork *work, Result &result) const noexcept override;

    size_t GetSize(MountWork *mountWork, Result &result, const PathView &path) const noexcept override;
    Result MakeDirectory(MountWork *mountWork, const PathView &path) noexcept override;
    Result Move(MountWork *mountWork, const PathView &sourcePath, const PathView &destPath) noexcept override;
    Result Remove(MountWork *mountWork, const PathView &path) noexcept override;
    bool Exists(MountWork *mountWork, Result &result, const PathView &path) noexcept override;

    //! このデリゲート用の作業用構造体
    struct Win32Work : public AccessWork
    {
        FILE *fp{nullptr};    //!< ファイルポインタ
        size_t size{0};    //!< ファイルサイズ
        
        //! 初期化用コンストラクタ
        Win32Work(OpenMode mode) noexcept
            : AccessWork(mode, kDelegateKey)
        {}
    };

private:
    [[nodiscard]] static Win32Work *GetWork(AccessWork *work, OpenMode mode = OpenMode::None) noexcept;
    [[nodiscard]] static const wchar_t *GetModeString(OpenMode mode) noexcept;
    [[nodiscard]] static STL::wstring ConvertPath(const PathView &path) noexcept;
    [[nodiscard]] static Error GetError(int errorNumber) noexcept;
};
}   // namespace MGL::File

#endif  // MGL_FILE_DELEGATE_ENABLE_WIN32
#endif	// INCGUARD_MGL_FILE_DELEGATE_WIN32_H_1616786750

// vim: et ts=4 sw=4 sts=4
