/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_file_delegate_nsfile.h
 *  \brief      MGL ファイルデリゲート Apple環境向け
 *  \date       Since: August 7, 2023. 15:48:36 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_FILE_DELEGATE_NSFILE_H_1691390916
#define INCGUARD_MGL_FILE_DELEGATE_NSFILE_H_1691390916

#include <mgl/mgl_environment.h>
#if defined(MGL_FILE_DELEGATE_ENABLE_NSFILE)

#include <stdio.h>

#include <mgl/file/mgl_file_delegate.h>

#if defined(__OBJC__)
#import <Foundation/NSError.h>
#endif

namespace MGL::File
{
// C++からはNSErrorを参照できないので前方宣言で誤魔化す
#if !defined(__OBJC__)
class NSError;
#endif

//! NSFile準拠のファイルデリゲート
class NSFileDelegate : public Delegate
{
public:
    //! このデリゲートを表すデリゲートキー
    inline static constexpr DelegateKey kDelegateKey = MakeDelegateKey("MGL-NSFile");
    
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このデリゲートがシステム標準のファイルを扱うものかを取得
     *  \retval     true    システム標準のファイルを扱う
     *  \retval     false   システム標準ではないファイルを扱う
     */
    /* ------------------------------------------------------------------------- */
    virtual bool IsManagedSystemNativeFile() const noexcept override
    {
        return true;    // 扱う
    }
        
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      このデリゲートが書き込みに対応しているかを取得
     *  \retval     true    対応している
     *  \retval     false   対応していない
     */
    /* ------------------------------------------------------------------------- */
    virtual bool IsWritable() const noexcept override
    {
        return true;    // 対応している
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウント時の処理
     *  \param[out] mountWork   マウント毎に使用するワークの共有ポインタ
     *  \param[in]  path        マウントパス
     *  \param[in]  accessType  アクセスタイプ
     *  \return     エラー発生時にいずれかのエラーが設定される
     */
    /* ------------------------------------------------------------------------- */
    virtual Result Mount([[maybe_unused]] SharedMountWork &mountWork, [[maybe_unused]] const PathView &path, [[maybe_unused]] MountAccessType accessType) noexcept override
    {
        return Result::Succeeded();     // このデリゲートはマウントワークを使用しないので何もせずに成功を返す
    }
    
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マウント解除時の処理
     *  \param[out] mountWork   マウント毎に使用するワークの共有ポインタ
     */
    /* ------------------------------------------------------------------------- */
    virtual void Unmount([[maybe_unused]] SharedMountWork &mountWork) noexcept override
    {
        return;
    }
    
    virtual AccessWorkPtr Open(Result &result, SharedMountWork &mountWork, const PathView &path, OpenMode mode) noexcept override;
    virtual Result Close(AccessWork *work) noexcept override;
    virtual size_t Read(AccessWork *work, Result &result, void *buffer, size_t size) noexcept override;
    virtual size_t Write(AccessWork *work, Result &result, const void *buffer, size_t size) noexcept override;
    virtual size_t Seek(AccessWork *work, Result &result, SeekType seekType, int32_t offset) noexcept override;
    virtual size_t GetOffset(AccessWork *work, Result &result) const noexcept override;
    virtual bool IsEOF(AccessWork *work, Result &result) const noexcept override;
    virtual size_t GetSize(AccessWork *work, Result &result) const noexcept override;
    virtual size_t GetSize(MountWork *mountWork, Result &result, const PathView &path) const noexcept override;
    virtual Result MakeDirectory(MountWork *mountWork, const PathView &path) noexcept override;
    virtual Result Move(MountWork *mountWork, const PathView &sourcePath, const PathView &destPath) noexcept override;
    virtual Result Remove(MountWork *mountWork, const PathView &path) noexcept override;
    virtual bool Exists(MountWork *mountWork, Result &result, const PathView &path) noexcept override;

    //! このデリゲート用の作業用構造体
    struct NSFileWork : public AccessWork
    {
        FILE *fp;       //!< ファイルポインタ
        size_t size;    //!< ファイルサイズ
        
        //! 初期化用コンストラクタ
        NSFileWork(OpenMode mode) noexcept
            : AccessWork(mode, kDelegateKey)
            , fp(nullptr)
            , size(0)
        {}        
    };

protected:
    NSFileWork *GetWork(AccessWork *work, OpenMode mode = OpenMode::None) const noexcept;
    const char *GetModeString(OpenMode mode) const noexcept;
    Error GetError(int errorNumber) const noexcept;
    Error GetError(const NSError *nsError) const noexcept;
};
}   // namespace MGL::File

#endif  // MGL_FILE_DELEGATE_ENABLE_NSFILE
#endif	// INCGUARD_MGL_FILE_DELEGATE_NSFILE_H_1691390916

// vim: et ts=4 sw=4 sts=4
