// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_event_types.h
 *  \brief      MGL イベント 型定義
 *  \date       Since: December 23, 2020. 13:26:23 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_EVENT_TYPES_H_1608697583
#define INCGUARD_MGL_EVENT_TYPES_H_1608697583

#include <cstddef>
#include <cstdint>

namespace MGL::Event
{
//! 通知タイプ
enum class NotifyType : uint8_t
{
    DidInitializableRenderer,       //!< レンダラが初期化可能になった際の通知

    AppInitialize,                  //!< アプリケーションの初期化
    AppFrameUpdate,                 //!< アプリケーションのフレーム更新
    AppExit,                        //!< アプリケーションの終了
    AppSuspend,                     //!< アプリケーションの中断

    PreFrameUpdate,                 //!< アプリケーションのフレーム更新前の処理
    PostFrameUpdate,                //!< アプリケーションのフレーム更新後の処理

    InputKeyDown,                   //!< キーボードのキーダウンイベント
    InputKeyUp,                     //!< キーボードのキーアップイベント
    ConnectKeyboard,                //!< キーボードの接続
    DisconnectKeyboard,             //!< キーボードの切断

    InputMouse,                     //!< マウス入力イベント

    InputTouch,                     //!< タッチ入力イベント

    ShouldClearInput,               //!< 入力クリア要求（主にmacOSのイベント非通知対策）

    DisconnectGamepad,              //!< エントリー済みのゲームパッドの切断

    ChangeClientSize,               //!< クライアントサイズの変更

    AudioSuspend,                   //!< オーディオの中断
    AudioResume,                    //!< オーディオの再開

    ChangeLanguage,                 //!< 設定言語の変更

    DeviceArrival,                  //!< デバイス接続
    DeviceRemove,                   //!< デバイス切断

    // 以下予約
    Reserve_Start,                             //!< 予約開始位置
    Reserve_ValidTypeCount = Reserve_Start,    //!< 有効な通知タイプの最大数
    Reserve_Invalid                            //!< 無効な通知タイプ
};

//! 有効な通知タイプの数
constexpr size_t kValidNotifyTypeCount = static_cast<size_t>(NotifyType::Reserve_ValidTypeCount);

//! イベント通知のユニークID型
using UniqueID = uintptr_t;

//! イベントコールバック関数
using CallbackFunction = void (*)(void *callbackArg, void *notifyArg);

}    // namespace MGL::Event
#endif    // INCGUARD_MGL_EVENT_TYPES_H_1608697583

// vim: et ts=4 sw=4 sts=4
