// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_event_register_list.h
 *  \brief      MGL イベント登録リスト
 *  \date       Since: December 23, 2020. 13:51:10 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_EVENT_REGISTER_LIST_H_1608699070
#define INCGUARD_MGL_EVENT_REGISTER_LIST_H_1608699070

#include <condition_variable>
#include <mutex>
#include <thread>

#include <mgl/event/mgl_event_handle.h>
#include <mgl/event/mgl_event_types.h>
#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/stl/mgl_stl_memory.h>

namespace MGL::Event
{
//! 登録アイテム
struct RegisterItem
{
    UniqueID id{0};                                  //!< 識別子
    NotifyType type{NotifyType::Reserve_Invalid};    //!< 通知タイプ
    CallbackFunction callback{nullptr};              //!< コールバック関数
    void *callbackArg{nullptr};                      //!< コールバック関数に渡す引数
};

//! 登録リスト
class RegisterList
{
public:
    RegisterList() noexcept;

    bool Register(STL::unique_ptr<RegisterItem> item) noexcept;
    void Unregister(Handle &handle) noexcept;

    bool Execute(void *notifyArg) noexcept;

private:
    using ItemList = STL::list<STL::unique_ptr<RegisterItem>>;

    bool IsRunning() const noexcept;
    bool IsRunning(UniqueID id) const noexcept;
    bool IsSameThread() const noexcept;

    void ApplyTemporaryRegisterList() noexcept;
    void ApplyTemporaryUnregisterList() noexcept;
    bool IsRequestedUnregister(UniqueID id) noexcept;

    ItemList _activeList;
    ItemList _temporaryRegisterList;
    STL::list<Handle> _temporaryUnregisterList;

    std::atomic<std::thread::id> _currentThreadId;
    std::mutex _activeListMutex;
    std::mutex _temporaryListMutex;

    std::condition_variable _syncExecute;
    std::mutex _syncExecuteMutex;
    UniqueID _executeUniqueId{0};
};

}    // namespace MGL::Event

#endif    // INCGUARD_MGL_EVENT_REGISTER_LIST_H_1608699070

// vim: et ts=4 sw=4 sts=4
