// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_event_handle.h
 *  \brief      MGL イベントハンドル
 *  \date       Since: December 23, 2020. 13:29:33 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_EVENT_HANDLE_H_1608697773
#define INCGUARD_MGL_EVENT_HANDLE_H_1608697773

#include <mgl/event/mgl_event_types.h>

namespace MGL::Event
{

//! イベントハンドルクラス
class Handle
{
public:
    Handle() noexcept;
    Handle(NotifyType type, CallbackFunction callback, void *callbackArg) noexcept;
    ~Handle() noexcept;

    void Set(NotifyType type, UniqueID id) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      通知タイプの取得
     *  \return     通知タイプ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr NotifyType GetType() const noexcept
    {
        return _type;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ユニークIDを取得
     *  \return     ユニークID
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr UniqueID GetUniqueID() const noexcept
    {
        return _id;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ハンドルの有効状態を取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsValid() const noexcept
    {
        return ((_type < NotifyType::Reserve_Start) && (_id != 0));
    }

    void Unregister() noexcept;

    // コピー禁止
    Handle(const Handle &) = delete;
    Handle &operator=(const Handle &) = delete;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ムーブコンストラクタ
     *  \param[in]  other   移動元ハンドル
     */
    /* ------------------------------------------------------------------------- */
    constexpr Handle(Handle &&other) noexcept
        : _type(other._type)
        , _id(other._id)
    {
        other._type = NotifyType::Reserve_Invalid;
        other._id = 0;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ムーブ代入演算
     *  \param[in]  other   移動元ハンドル
     */
    /* ------------------------------------------------------------------------- */
    constexpr Handle &operator=(Handle &&other) noexcept
    {
        if (this != &other)
        {
            Unregister();

            _type = other._type;
            _id = other._id;

            other._type = NotifyType::Reserve_Invalid;
            other._id = 0;
        }

        return *this;
    }

private:
    NotifyType _type;
    UniqueID _id;
};

}    // namespace MGL::Event
#endif    // INCGUARD_MGL_EVENT_HANDLE_H_1608697773

// vim: et ts=4 sw=4 sts=4
