// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_unique_ptr.h
 *  \brief      MGL ユニークポインタ
 *  \date       Since: April 5, 2025. 22:01:38 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_UNIQUE_PTR_H_1743858098
#define INCGUARD_MGL_UNIQUE_PTR_H_1743858098

#include <memory>
#include <type_traits>

#include <mgl/memory/mgl_memory.h>
#include <mgl/stl/mgl_stl_memory.h>

namespace MGL
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ユニークポインタ
 *  \tparam     T   扱う型の指定。配列には非対応。
 */
/* ------------------------------------------------------------------------- */
template <class T>
class UniquePtr
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr UniquePtr() noexcept
        : _ptr(nullptr)
    {
    }

    // コピーは禁止
    UniquePtr(UniquePtr &) = delete;
    UniquePtr &operator=(UniquePtr &) = delete;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ムーブコンストラクタ
     *  \param[in]  rhs     右辺値
     */
    /* ------------------------------------------------------------------------- */
    UniquePtr(UniquePtr &&rhs) noexcept
    {
        Delete();
        _ptr = rhs._ptr;    // NOLINT(cppcoreguidelines-prefer-member-initializer) : Delete()が呼べなくなるので無理
        rhs._ptr = nullptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      演算子によるムーブ
     *  \param[in]  rhs     右辺値
     */
    /* ------------------------------------------------------------------------- */
    UniquePtr &operator=(UniquePtr &&rhs) noexcept
    {
        Delete();
        _ptr = rhs._ptr;
        rhs._ptr = nullptr;
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デストラクタ
     */
    /* ------------------------------------------------------------------------- */
    ~UniquePtr() noexcept
    {
        Delete();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      オブジェクトを生成
     *  \tparam     U       生成するオブジェクトの型。省略時はT型として扱われる。
     *  \param[in]  args    コンストラクタに渡す引数
     *  \return     自分自身の参照
     */
    /* ------------------------------------------------------------------------- */
    template <class U = T, class... Args>
    UniquePtr &New(Args &&...args) noexcept
    {
        // TとUが異なる型、かつ継承関係に無い場合は生成できない
        if constexpr (std::is_class_v<T> && !std::is_same_v<T, U>)
        {
            static_assert(std::is_base_of_v<T, U>, "Cannot be converted to the specified type.");
        }

        Delete();
        _ptr = Allocate<U>(std::forward<Args>(args)...);
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      初期化付きのクラスオブジェクトの生成
     *  \tparam     U           生成するオブジェクトの型。省略時はT型として扱われる。
     *  \param[in]  initializer 初期化用の関数
     *  \param[in]  args        コンストラクタに渡す引数
     *  \return     自分自身の参照
     */
    /* ------------------------------------------------------------------------- */
    template <class U = T, class Initializer, class... Args>
    UniquePtr &InitNew(Initializer initializer, Args &&...args) noexcept
    {
        // TとUが異なる型、かつ継承関係に無い場合は生成できない
        if constexpr (std::is_class_v<T> && !std::is_same_v<T, U>)
        {
            static_assert(std::is_base_of_v<T, U>, "Cannot be converted to the specified type.");
        }

        // 既存のオブジェクトを削除して再生成
        Delete();
        auto obj = Allocate<U>(std::forward<Args>(args)...);

        // オブジェクトが生成されていれば初期化関数を呼び出し
        if (obj != nullptr)
        {
            if (!InvokeInitializer(initializer, *obj))
            {
                std::destroy_at(obj);
                Memory::Deallocate(obj);
                obj = nullptr;
            }
        }
        _ptr = obj;
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      オブジェクトの削除
     */
    /* ------------------------------------------------------------------------- */
    void Delete() noexcept
    {
        if (_ptr != nullptr)
        {
            std::destroy_at(_ptr);
            Memory::Deallocate(_ptr);
            _ptr = nullptr;
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ポインタを取得
     *  \return     保持しているポインタ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr T *Get() const noexcept
    {
        return _ptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      演算子によるポインタへのアクセス
     *  \return     保持しているポインタ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr T *operator->() const noexcept
    {
        return _ptr;
    }


    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      演算子によるポインタの間接参照
     *  \return     ポインタの参照先
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr std::add_lvalue_reference_t<T> operator*() const noexcept
    {
        return *_ptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      保持しているポインタがNULLであるかをチェック
     *  \retval     true    ポインタがNULL
     *  \retval     false   ポインタがNULLではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsNull() const noexcept
    {
        return _ptr == nullptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      保持しているポインタが無効であるかを取得
     *  \retval     true    無効
     *  \retval     false   有効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool operator!() const noexcept
    {
        return _ptr == nullptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      保持しているポインタが有効であるかを取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] explicit constexpr operator bool() const noexcept
    {
        return _ptr != nullptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      新たなクラスオブジェクトの生成
     *  \tparam     U       生成するクラスの型
     *  \param[in]  args    コンストラクタに渡す引数
     *  \return     新たに生成されたユニークポインタ
     */
    /* ------------------------------------------------------------------------- */
    template <class U = T, class... Args>
    [[nodiscard]] static UniquePtr Make(Args &&...args) noexcept
    {
        // TとUが異なる型、かつ継承関係に無い場合は生成できない
        if constexpr (std::is_class_v<T> && !std::is_same_v<T, U>)
        {
            static_assert(std::is_base_of_v<T, U>, "Cannot be converted to the specified type.");
        }

        return UniquePtr(Allocate<U>(std::forward<Args>(args)...));
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      新たなクラスオブジェクトの生成
     *  \tparam     U           生成するクラスの型
     *  \param[in]  initializer 初期化処理の関数オブジェクト
     *  \param[in]  args        コンストラクタに渡す引数
     *  \return     新たに生成されたユニークポインタ
     */
    /* ------------------------------------------------------------------------- */
    template <class U = T, class Initializer, class... Args>
    [[nodiscard]] static UniquePtr InitMake(Initializer initializer, Args &&...args) noexcept
    {
        // TとUが異なる型、かつ継承関係に無い場合は生成できない
        if constexpr (std::is_class_v<T> && !std::is_same_v<T, U>)
        {
            static_assert(std::is_base_of_v<T, U>, "Cannot be converted to the specified type.");
        }

        // オブジェクトを生成して初期化関数を呼び出し
        auto obj = Allocate<U>(std::forward<Args>(args)...);
        if (obj != nullptr)
        {
            if (!InvokeInitializer(initializer, *obj))
            {
                std::destroy_at(obj);
                Memory::Deallocate(obj);
                obj = nullptr;
            }
        }

        return UniquePtr(obj);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      シェアードポインタへのムーブ
     *  \return     シェアードポインタ
     */
    /* ------------------------------------------------------------------------- */
    std::shared_ptr<T> MoveSharedPtr()
    {
        // シェアードポインタへ渡すデリータ
        auto deleter = [](T *ptr) constexpr
        {
            if (ptr != nullptr)
            {
                std::destroy_at(ptr);
                Memory::Deallocate(ptr);
            }
        };

        // シェアードポインタに渡すアロケータ
        const STL::Allocator<void> allocator;

        // デリータとアロケータを使用するシェアードポインタに渡して、自身はnullptrに
        auto sharedPtr = std::shared_ptr<T>(_ptr, deleter, allocator);
        _ptr = nullptr;

        return sharedPtr;
    }

private:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ポインタ初期化用のコンストラクタ（内部呼び出し用）
     *  \param[in]  ptr     設定するポインタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr UniquePtr(T *ptr) noexcept
        : _ptr(ptr)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      新たなオブジェクトの生成
     *  \param[in]  args    コンストラクタに与える引数
     *  \return     生成したクラスオブジェクト
     */
    /* ------------------------------------------------------------------------- */
    template <class U, class... Args>
    [[nodiscard]] static U *Allocate(Args &&...args) noexcept
    {
        // クラスオブジェクトを生成
        auto *ptr = Memory::Allocate(sizeof(U));
        if (ptr == nullptr)
        {
            return nullptr;
        }

        // 例外なし
        if constexpr (std::is_nothrow_constructible_v<U, Args...>)
        {
            return new (ptr) U(std::forward<Args>(args)...);
        }
        // 例外あり
        else
        {
            try
            {
                return new (ptr) U(std::forward<Args>(args)...);
            }
            catch (...)
            {
                Memory::Deallocate(ptr);
            }
        }

        return nullptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      初期化関数の呼び出し
     *  \tparam     U           生成するオブジェクトの型
     *  \param[in]  initializer 初期化用の関数
     *  \param[in]  obj         初期化するオブジェクト
     *  \retval     true        成功
     *  \retval     false       失敗
     */
    /* ------------------------------------------------------------------------- */
    template <class U, class Initializer>
    static bool InvokeInitializer(Initializer initializer, U &obj) noexcept
    {
        // 関数がbool型の戻り値を返す場合はその結果を返す
        if constexpr (std::is_invocable_r_v<bool, decltype(initializer), U &>)
        {
            return initializer(obj);
        }
        // 関数の戻り値がvoidなら常にtrue
        else if constexpr (std::is_invocable_r_v<void, decltype(initializer), U &>)
        {
            initializer(obj);
            return true;
        }
        // それ以外は不適格
        else
        {
            static_assert(std::false_type::value, "Initializer must be U& type argument and bool or void value returns.");
            return false;
        }
    }

    //! オブジェクトへのポインタ
    T *_ptr{nullptr};
};


// 配列は扱えない
template <class T>
class UniquePtr<T[]>
{
    static_assert(false, "MGL::UniquePtr cannot specification array type.");
};

// サイズ付きの配列も扱えない
template <class T, size_t n>
class UniquePtr<T[n]>
{
    static_assert(false, "MGL::UniquePtr cannot specification sized array type.");
};
}    // namespace MGL

#endif    // INCGUARD_MGL_UNIQUE_PTR_H_1743858098

// vim: et ts=4 sw=4 sts=4
