// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_singleton.h
 *  \brief      MGL シングルトンクラス
 *  \date       Since: December 16, 2020. 17:06:10 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_SINGLETON_H_1608105970
#define INCGUARD_MGL_SINGLETON_H_1608105970

#include <cassert>
#include <mutex>

#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/stl/mgl_stl_memory.h>

namespace MGL
{
//! シングルトン解放クラス
class SingletonFinalizer
{
public:
    //! 解放用の関数の型
    using FinalizeFunction = void (*)();

    static bool Initialize() noexcept;

    static void Register(FinalizeFunction function) noexcept;
    static void Finalize() noexcept;
    static void Unregister(FinalizeFunction function) noexcept;

private:
    static inline std::list<FinalizeFunction> _functions;    // staticなのでMGL::STL::listは使用できない
    static inline std::unique_ptr<std::recursive_mutex> _mutex;
};

//! シングルトンテンプレート（共有ライブラリ用）
template<typename T, class... Args>
class SharedSingleton
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インスタンスの生成
     *  \return     生成したインスタンスの参照
     */
    /* ------------------------------------------------------------------------- */
    static T &CreateInstance(Args... args) noexcept
    {
        auto &instance = T::GetInstanceRef();

        if (instance.get() == nullptr)
        {
            instance = STL::make_unique<T>(args...);
            if (instance.get() != nullptr)
            {
                SingletonFinalizer::Register(DestroyInstance);
            }
        }

        return *(instance.get());
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インスタンスの取得
     *  \return     インスタンスの参照
     */
    /* ------------------------------------------------------------------------- */
    static T &GetInstance() noexcept
    {
        auto &instance = T::GetInstanceRef();
        assert(instance.get());
        return *(instance.get());
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インスタンスの破棄
     */
    /* ------------------------------------------------------------------------- */
    static void DestroyInstance() noexcept
    {
        if (HasInstance())
        {
            SingletonFinalizer::Unregister(DestroyInstance);
            T::GetInstanceRef().reset();
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効なインスタンスを保持しているかを取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    static bool HasInstance() noexcept
    {
        return T::GetInstanceRef().get() != nullptr;
    }

    // コピーとムーブは禁止
    SharedSingleton(const SharedSingleton &) noexcept = delete;
    SharedSingleton &operator=(const SharedSingleton &) noexcept = delete;
    SharedSingleton(SharedSingleton &&) noexcept = delete;
    SharedSingleton &operator=(SharedSingleton &&) noexcept = delete;

    virtual ~SharedSingleton() noexcept = default;
    
protected:
    // 外から作れないようコンストラクタはprotectedに定義
    SharedSingleton() noexcept = default;
};


//! シングルトンテンプレート（アプリケーションまたは静的リンクライブラリのみ利用可能）
template<typename T, class... Args>
class StaticSingleton
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インスタンスの生成
     *  \return     生成したインスタンスの参照
     */
    /* ------------------------------------------------------------------------- */
    static T &CreateInstance(Args... args) noexcept
    {
        if (!HasInstance())
        {
            _instance = STL::make_unique<T>(args...);
            if (HasInstance())
            {
                SingletonFinalizer::Register(DestroyInstance);
            }
        }

        return *_instance;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インスタンスの取得
     *  \return     インスタンスの参照
     */
    /* ------------------------------------------------------------------------- */
    static constexpr T &GetInstance() noexcept
    {
        assert(_instance);
        return *_instance;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インスタンスの破棄
     */
    /* ------------------------------------------------------------------------- */
    static void DestroyInstance() noexcept
    {
        if (HasInstance())
        {
            SingletonFinalizer::Unregister(DestroyInstance);
            _instance.reset();
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      有効なインスタンスを保持しているかを取得
     *  \retval     true    有効
     *  \retval     false   無効
     */
    /* ------------------------------------------------------------------------- */
    static constexpr bool HasInstance() noexcept
    {
        return _instance != nullptr;
    }

    // コピーとムーブは禁止
    StaticSingleton(const StaticSingleton &) noexcept = delete;
    StaticSingleton &operator=(const StaticSingleton &) noexcept = delete;
    StaticSingleton(StaticSingleton &&) noexcept = delete;
    StaticSingleton &operator=(StaticSingleton &&) noexcept = delete;

    virtual ~StaticSingleton() noexcept = default;
    
protected:
    // 外から作れないようコンストラクタはprotectedに定義
    StaticSingleton() noexcept = default;

private:
    static inline STL::unique_ptr<T> _instance = nullptr;
};

}    // namespace MGL
#endif    // INCGUARD_MGL_SINGLETON_H_1608105970

// vim: et ts=4 sw=4 sts=4
