// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_result_wrapper.h
 *  \brief      MGL エラーを処理結果としてラップするためのテンプレート
 *  \date       Since: January 20, 2021. 20:02:10 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_RESULT_WRAPPER_H_1611140530
#define INCGUARD_MGL_RESULT_WRAPPER_H_1611140530

#include <type_traits>

namespace MGL
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      エラーをラップするテンプレートクラス
 *  \tparam     ErrorType           エラーを表す型
 *  \tparam     noErrorValue        ErrorTypeで指定した型で，エラーが発生していない事を表す値
 *  \tparam     defaultErrorValue   クラス生成時の初期値．省略時はnoErrorValueと同じ
 */
/* ------------------------------------------------------------------------- */
template <typename ErrorType, ErrorType noErrorValue = ErrorType(), ErrorType defaultErrorValue = noErrorValue>
class ResultWrapper
{
public:
    static constexpr ErrorType kNoError = noErrorValue;              //!< エラーが発生していない事を表す値
    static constexpr ErrorType kDefaultError = defaultErrorValue;    //!< エラーの初期値

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラー設定用コンストラクタ
     *  \param[in]  error   設定するエラー
     */
    /* ------------------------------------------------------------------------- */
    constexpr ResultWrapper(ErrorType error) noexcept
        : _error(error)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr ResultWrapper() noexcept
        : _error(defaultErrorValue)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラーを取得
     *  \return     設定されているエラー
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr ErrorType GetError() const noexcept
    {
        return _error;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラーコードを取得
     *  \return     設定されているエラーを基底型にキャストした値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr std::underlying_type_t<ErrorType> GetErrorCode() const noexcept
    {
        return static_cast<std::underlying_type_t<ErrorType>>(_error);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラーを設定
     *  \param[in]  error   設定するエラー
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetError(ErrorType error) noexcept
    {
        _error = error;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラーを無効化
     */
    /* ------------------------------------------------------------------------- */
    constexpr void Success() noexcept
    {
        _error = noErrorValue;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラーのないオブジェクトを取得
     *  \return     エラーのないオブジェクト
     */
    /* ------------------------------------------------------------------------- */
    static constexpr ResultWrapper Succeeded() noexcept
    {
        return ResultWrapper(noErrorValue);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラーの代入演算子
     *  \param[in]  rhs     代入するエラー
     */
    /* ------------------------------------------------------------------------- */
    constexpr ResultWrapper &operator=(const ErrorType &rhs) noexcept
    {
        _error = rhs;
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラーの等価演算子
     *  \param[in]  rhs     比較するエラー
     *  \retval     true    等価
     *  \retval     false   不等価
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool operator==(const ErrorType &rhs) const noexcept
    {
        return _error == rhs;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラーの不等価演算子
     *  \param[in]  rhs     比較するエラー
     *  \retval     true    不等価
     *  \retval     false   等価
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool operator!=(const ErrorType &rhs) const noexcept
    {
        return _error != rhs;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラーが発生しているかを取得
     *  \retval     true    エラーが発生している
     *  \retval     false   エラーは発生していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool HasError() const noexcept
    {
        return _error != noErrorValue;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      エラーが発生していないかを取得
     *  \retval     true    エラーは発生していない
     *  \retval     false   エラーが発生している
     *  \note       HasError()と逆の結果を返す
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsSucceeded() const noexcept
    {
        return !HasError();
    }

private:
    ErrorType _error;
};
}    // namespace MGL

#endif    // INCGUARD_MGL_RESULT_WRAPPER_H_1611140530

// vim: et ts=4 sw=4 sts=4
