// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_range.h
 *  \brief      MGL 値の範囲を表現するクラス
 *  \date       Since: May 11, 2025. 1:49:29 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_RANGE_H_1746895769
#define INCGUARD_MGL_RANGE_H_1746895769

#include <algorithm>
#include <limits>
#include <type_traits>

namespace MGL
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      値の範囲を表現するクラス
 *  \tparam     T   扱う値の型
 */
/* ------------------------------------------------------------------------- */
template <class T>
class Range
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr Range() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  begin   開始値
     *  \param[in]  end     終了値
     */
    /* ------------------------------------------------------------------------- */
    constexpr Range(T begin, T end) noexcept
        : _begin(begin)
        , _end(end)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      開始値を取得
     *  \return     開始値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr T Begin() const noexcept
    {
        return _begin;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      終了値を取得
     *  \return     終了値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr T End() const noexcept
    {
        return _end;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      値が正の方向に進むかを取得
     *  \retval     true    値は正の方向に進む
     *  \retval     false   値は負の方向に進む
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsPositive() const noexcept
    {
        return _begin < _end;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      値が負の方向に進むかを取得
     *  \retval     true    値は負の方向に進む
     *  \retval     false   値は正の方向に進む
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsNegative() const noexcept
    {
        return !IsPositive();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      値が範囲を持たない単一の値であるかを取得
     *  \retval     true    値は範囲を持たない
     *  \retval     false   値は範囲を持つ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsSingleValue() const noexcept
    {
        if constexpr (std::is_floating_point_v<T>)
        {
            return _end - _begin <= std::numeric_limits<T>::epsilon();
        }
        else
        {
            return _begin == _end;
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      範囲がゼロであるかを取得
     *  \retval     true    範囲がゼロ
     *  \retval     false   範囲がゼロではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsZero() const noexcept
    {
        if constexpr (std::is_floating_point_v<T>)
        {
            return std::abs(_begin) <= std::numeric_limits<T>::epsilon() &&
                   std::abs(_end) <= std::numeric_limits<T>::epsilon();
        }
        else
        {
            return _begin == 0 && _end == 0;
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定された値の次の値を取得
     *  \param[in]  value   現在の値
     *  \return     valueの次の値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr T Next(T value) const noexcept
    {
        if (IsPositive() || IsSingleValue())
        {
            return ++value;
        }
        else
        {
            return --value;
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定された値の前の値を取得
     *  \param[in]  value   現在の値
     *  \return     valueの前の値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr T Previous(T value) const noexcept
    {
        if (IsPositive() || IsSingleValue())
        {
            return --value;
        }
        else
        {
            return ++value;
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定された値が範囲内に含まれているかを取得
     *  \param[in]  value   指定値
     *  \retval     true    含まれている
     *  \retval     false   含まれていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool Contains(T value) const noexcept
    {
        if (IsPositive() || IsSingleValue())
        {
            return (value >= _begin) && (value <= _end);
        }
        else
        {
            return (value <= _begin) && (value >= _end);
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定した値を範囲内にクランプ
     *  \param[in]  value   指定値
     *  \return     クランプされた値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr T Clamp(T value) const noexcept
    {
        if (IsSingleValue())
        {
            return _begin;
        }
        else if (IsPositive())
        {
            return std::clamp(value, _begin, _end);
        }
        else
        {
            return std::clamp(value, _end, _begin);
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      値を0.0fから1.0fの値に正規化
     *  \param[in]  value   正規化する値
     *  \return     正規化された値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr float Normalize(T value) const noexcept
    {
        if (IsSingleValue())
        {
            return 1.0f;
        }

        const auto fBegin = static_cast<float>(_begin);
        const auto fEnd = static_cast<float>(_end);
        const auto fValue = static_cast<float>(value);

        return (fValue - fBegin) / (fEnd - fBegin);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      正規化された値からスケーリング
     *  \param[in]  normalizedValue   正規化された値
     *  \return     スケーリングされた値
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr T Scale(float normalizedValue) const noexcept
    {
        const auto fBegin = static_cast<float>(_begin);
        const auto fEnd = static_cast<float>(_end);

        return static_cast<T>(fBegin + ((fEnd - fBegin) * normalizedValue));
    }

private:
    T _begin{0};
    T _end{0};
};
}    // namespace MGL

#endif    // INCGUARD_MGL_RANGE_H_1746895769

// vim: et ts=4 sw=4 sts=4
