// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_color_xterm.h
 *  \brief      MGL xterm-256カラー定義
 *  \date       Since: March 11, 2023. 0:13:31 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_COLOR_XTERM_H_1678461211
#define INCGUARD_MGL_COLOR_XTERM_H_1678461211

#include <cstddef>
#include <cstdint>
#include <optional>

namespace MGL
{
//! xterm 256カラー
enum class XColor : uint8_t
{
    Black,      // 0
    Maroon,     // 1
    Green,      // 2
    Olive,      // 3
    Navy,       // 4
    Purple,     // 5
    Teal,       // 6
    Silver,     // 7
    Gray,       // 8
    Red,        // 9
    Lime,       // 10
    Yellow,     // 11
    Blue,       // 12
    Magenta,    // 13
    Cyan,       // 14
    White,      // 15

    C000, C001, C002, C003, C004, C005,     // 16-21
    C010, C011, C012, C013, C014, C015,     // 22-27
    C020, C021, C022, C023, C024, C025,     // 28-33
    C030, C031, C032, C033, C034, C035,     // 34-39
    C040, C041, C042, C043, C044, C045,     // 40-45
    C050, C051, C052, C053, C054, C055,     // 46-51

    C100, C101, C102, C103, C104, C105,     // 52-57
    C110, C111, C112, C113, C114, C115,     // 58-63
    C120, C121, C122, C123, C124, C125,     // 64-69
    C130, C131, C132, C133, C134, C135,     // 70-75
    C140, C141, C142, C143, C144, C145,     // 76-81
    C150, C151, C152, C153, C154, C155,     // 82-87

    C200, C201, C202, C203, C204, C205,     // 88-93
    C210, C211, C212, C213, C214, C215,     // 94-99
    C220, C221, C222, C223, C224, C225,     // 100-105
    C230, C231, C232, C233, C234, C235,     // 106-111
    C240, C241, C242, C243, C244, C245,     // 112-117
    C250, C251, C252, C253, C254, C255,     // 118-123

    C300, C301, C302, C303, C304, C305,     // 124-129
    C310, C311, C312, C313, C314, C315,     // 130-135
    C320, C321, C322, C323, C324, C325,     // 136-141
    C330, C331, C332, C333, C334, C335,     // 142-147
    C340, C341, C342, C343, C344, C345,     // 148-153
    C350, C351, C352, C353, C354, C355,     // 154-159

    C400, C401, C402, C403, C404, C405,     // 160-165
    C410, C411, C412, C413, C414, C415,     // 166-171
    C420, C421, C422, C423, C424, C425,     // 172-177
    C430, C431, C432, C433, C434, C435,     // 178-183
    C440, C441, C442, C443, C444, C445,     // 184-189
    C450, C451, C452, C453, C454, C455,     // 190-195

    C500, C501, C502, C503, C504, C505,     // 196-201
    C510, C511, C512, C513, C514, C515,     // 202-207
    C520, C521, C522, C523, C524, C525,     // 208-213
    C530, C531, C532, C533, C534, C535,     // 214-219
    C540, C541, C542, C543, C544, C545,     // 220-225
    C550, C551, C552, C553, C554, C555,     // 226-231

    // 232-255
	Gray0,  Gray1,  Gray2,  Gray3,  Gray4,  Gray5,
	Gray6,  Gray7,  Gray8,  Gray9,  Gray10, Gray11,
	Gray12, Gray13, Gray14, Gray15, Gray16, Gray17,
	Gray18, Gray19, Gray20, Gray21, Gray22, Gray23,
};
static_assert(XColor::Gray23 == XColor{255}, "XColor table is invalid.");


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      XColorのカラーコードを取得
 *  \param[in]  xcolor  xtermカラー
 *  \return     対応するカラーコード
 */
/* ------------------------------------------------------------------------- */
constexpr uint32_t GetXColorCode(XColor xcolor) noexcept
{
    constexpr const uint32_t kColorTable[] =
    {
        0x000000FF, 0x800000FF, 0x008000FF, 0x808000FF, 0x000080FF, 0x800080FF, 0x008080FF, 0xC0C0C0FF,
        0x808080FF, 0xFF0000FF, 0x00FF00FF, 0xFFFF00FF, 0x0000FFFF, 0xFF00FFFF, 0x00FFFFFF, 0xFFFFFFFF,
        0x000000FF, 0x00005FFF, 0x000087FF, 0x0000AFFF, 0x0000D7FF, 0x0000FFFF, 0x005F00FF, 0x005F5FFF,
        0x005F87FF, 0x005FAFFF, 0x005FD7FF, 0x005FFFFF, 0x008700FF, 0x00875FFF, 0x008787FF, 0x0087AFFF,
        0x0087D7FF, 0x0087FFFF, 0x00AF00FF, 0x00AF5FFF, 0x00AF87FF, 0x00AFAFFF, 0x00AFD7FF, 0x00AFFFFF,
        0x00D700FF, 0x00D75FFF, 0x00D787FF, 0x00D7AFFF, 0x00D7D7FF, 0x00D7FFFF, 0x00FF00FF, 0x00FF5FFF,
        0x00FF87FF, 0x00FFAFFF, 0x00FFD7FF, 0x00FFFFFF, 0x5F0000FF, 0x5F005FFF, 0x5F0087FF, 0x5F00AFFF,
        0x5F00D7FF, 0x5F00FFFF, 0x5F5F00FF, 0x5F5F5FFF, 0x5F5F87FF, 0x5F5FAFFF, 0x5F5FD7FF, 0x5F5FFFFF,
        0x5F8700FF, 0x5F875FFF, 0x5F8787FF, 0x5F87AFFF, 0x5F87D7FF, 0x5F87FFFF, 0x5FAF00FF, 0x5FAF5FFF,
        0x5FAF87FF, 0x5FAFAFFF, 0x5FAFD7FF, 0x5FAFFFFF, 0x5FD700FF, 0x5FD75FFF, 0x5FD787FF, 0x5FD7AFFF,
        0x5FD7D7FF, 0x5FD7FFFF, 0x5FFF00FF, 0x5FFF5FFF, 0x5FFF87FF, 0x5FFFAFFF, 0x5FFFD7FF, 0x5FFFFFFF,
        0x870000FF, 0x87005FFF, 0x870087FF, 0x8700AFFF, 0x8700D7FF, 0x8700FFFF, 0x875F00FF, 0x875F5FFF,
        0x875F87FF, 0x875FAFFF, 0x875FD7FF, 0x875FFFFF, 0x878700FF, 0x87875FFF, 0x878787FF, 0x8787AFFF,
        0x8787D7FF, 0x8787FFFF, 0x87AF00FF, 0x87AF5FFF, 0x87AF87FF, 0x87AFAFFF, 0x87AFD7FF, 0x87AFFFFF,
        0x87D700FF, 0x87D75FFF, 0x87D787FF, 0x87D7AFFF, 0x87D7D7FF, 0x87D7FFFF, 0x87FF00FF, 0x87FF5FFF,
        0x87FF87FF, 0x87FFAFFF, 0x87FFD7FF, 0x87FFFFFF, 0xAF0000FF, 0xAF005FFF, 0xAF0087FF, 0xAF00AFFF,
        0xAF00D7FF, 0xAF00FFFF, 0xAF5F00FF, 0xAF5F5FFF, 0xAF5F87FF, 0xAF5FAFFF, 0xAF5FD7FF, 0xAF5FFFFF,
        0xAF8700FF, 0xAF875FFF, 0xAF8787FF, 0xAF87AFFF, 0xAF87D7FF, 0xAF87FFFF, 0xAFAF00FF, 0xAFAF5FFF,
        0xAFAF87FF, 0xAFAFAFFF, 0xAFAFD7FF, 0xAFAFFFFF, 0xAFD700FF, 0xAFD75FFF, 0xAFD787FF, 0xAFD7AFFF,
        0xAFD7D7FF, 0xAFD7FFFF, 0xAFFF00FF, 0xAFFF5FFF, 0xAFFF87FF, 0xAFFFAFFF, 0xAFFFD7FF, 0xAFFFFFFF,
        0xD70000FF, 0xD7005FFF, 0xD70087FF, 0xD700AFFF, 0xD700D7FF, 0xD700FFFF, 0xD75F00FF, 0xD75F5FFF,
        0xD75F87FF, 0xD75FAFFF, 0xD75FD7FF, 0xD75FFFFF, 0xD78700FF, 0xD7875FFF, 0xD78787FF, 0xD787AFFF,
        0xD787D7FF, 0xD787FFFF, 0xD7AF00FF, 0xD7AF5FFF, 0xD7AF87FF, 0xD7AFAFFF, 0xD7AFD7FF, 0xD7AFFFFF,
        0xD7D700FF, 0xD7D75FFF, 0xD7D787FF, 0xD7D7AFFF, 0xD7D7D7FF, 0xD7D7FFFF, 0xD7FF00FF, 0xD7FF5FFF,
        0xD7FF87FF, 0xD7FFAFFF, 0xD7FFD7FF, 0xD7FFFFFF, 0xFF0000FF, 0xFF005FFF, 0xFF0087FF, 0xFF00AFFF,
        0xFF00D7FF, 0xFF00FFFF, 0xFF5F00FF, 0xFF5F5FFF, 0xFF5F87FF, 0xFF5FAFFF, 0xFF5FD7FF, 0xFF5FFFFF,
        0xFF8700FF, 0xFF875FFF, 0xFF8787FF, 0xFF87AFFF, 0xFF87D7FF, 0xFF87FFFF, 0xFFAF00FF, 0xFFAF5FFF,
        0xFFAF87FF, 0xFFAFAFFF, 0xFFAFD7FF, 0xFFAFFFFF, 0xFFD700FF, 0xFFD75FFF, 0xFFD787FF, 0xFFD7AFFF,
        0xFFD7D7FF, 0xFFD7FFFF, 0xFFFF00FF, 0xFFFF5FFF, 0xFFFF87FF, 0xFFFFAFFF, 0xFFFFD7FF, 0xFFFFFFFF,
        0x080808FF, 0x121212FF, 0x1C1C1CFF, 0x262626FF, 0x303030FF, 0x3A3A3AFF, 0x444444FF, 0x4E4E4EFF,
        0x585858FF, 0x626262FF, 0x6C6C6CFF, 0x767676FF, 0x808080FF, 0x8A8A8AFF, 0x949494FF, 0x9E9E9EFF,
        0xA8A8A8FF, 0xB2B2B2FF, 0xBCBCBCFF, 0xC6C6C6FF, 0xD0D0D0FF, 0xDADADAFF, 0xE4E4E4FF, 0xEEEEEEFF,        
    };

    return kColorTable[static_cast<size_t>(xcolor)];
}

std::optional<XColor> GetXColor(uint32_t hash) noexcept;
}    // namespace MGL

#endif	// INCGUARD_MGL_COLOR_XTERM_H_1678461211

// vim: et ts=4 sw=4 sts=4
