// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_color32.h
 *  \brief      32-bit色情報
 *  \date       Since: March 11, 2023. 2:38:32 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_COLOR32_H_1678469912
#define INCGUARD_MGL_COLOR32_H_1678469912

#include <mgl/common/mgl_color.h>

namespace MGL
{
//! 各8bit RGBAの色情報
struct Color32
{
    uint8_t red;      //!< 赤成分
    uint8_t green;    //!< 緑成分
    uint8_t blue;     //!< 青成分
    uint8_t alpha;    //!< アルファ値

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief  ゼロ初期化
     */
    /* ------------------------------------------------------------------------- */
    constexpr Color32() noexcept
        : red(0)
        , green(0)
        , blue(0)
        , alpha(0)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      色を指定して初期化
     *  \param[in]  inRed   赤
     *  \param[in]  inGreen 緑
     *  \param[in]  inBlue  青
     *  \param[in]  inAlpha アルファ値
     */
    /* ------------------------------------------------------------------------- */
    constexpr Color32(uint8_t inRed, uint8_t inGreen, uint8_t inBlue, uint8_t inAlpha = 255) noexcept
        : red(inRed)
        , green(inGreen)
        , blue(inBlue)
        , alpha(inAlpha)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      32-bit RGBA値で初期化
     *  \param[in]  rgba    RGBA値
     */
    /* ------------------------------------------------------------------------- */
    constexpr Color32(uint32_t rgba) noexcept
        : red(static_cast<uint8_t>(rgba >> 24))
        , green(static_cast<uint8_t>((rgba >> 16) & 0xFF))
        , blue((static_cast<uint8_t>(rgba >> 8) & 0xFF))
        , alpha(static_cast<uint8_t>(rgba & 0xFF))
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      xtermカラーで色を指定して初期化
     *  \param[in]  xcolor   xtermカラー
     */
    /* ------------------------------------------------------------------------- */
    constexpr Color32(XColor xcolor) noexcept
        : Color32(GetXColorCode(xcolor))
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief  floatの色情報にキャスト
     */
    /* ------------------------------------------------------------------------- */
    constexpr operator Color() const noexcept
    {
        Color color;
        color.red = static_cast<float>(red) / 255.0f;
        color.green = static_cast<float>(green) / 255.0f;
        color.blue = static_cast<float>(blue) / 255.0f;
        color.alpha = static_cast<float>(alpha) / 255.0f;
        return color;
    }
};

}    // namespace MGL
#endif    // INCGUARD_MGL_COLOR32_H_1678469912

// vim: et ts=4 sw=4 sts=4
