// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_color.h
 *  \brief      MGL 色情報
 *  \date       Since: November 29, 2020. 15:17:59 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_COLOR_H_1606630679
#define INCGUARD_MGL_COLOR_H_1606630679

#include <cstdint>

#include "mgl_color_xterm.h"

namespace MGL
{
//! 色情報
struct Color
{
    float red;      //!< 赤成分
    float green;    //!< 緑成分
    float blue;     //!< 青成分
    float alpha;    //!< アルファ値

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief  ゼロ初期化
     */
    /* ------------------------------------------------------------------------- */
    constexpr Color() noexcept
        : red(0.0f)
        , green(0.0f)
        , blue(0.0f)
        , alpha(0.0f)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      色を指定して初期化
     *  \param[in]  inRed   赤
     *  \param[in]  inGreen 緑
     *  \param[in]  inBlue  青
     *  \param[in]  inAlpha アルファ値
     */
    /* ------------------------------------------------------------------------- */
    constexpr Color(float inRed, float inGreen, float inBlue, float inAlpha = 1.0f) noexcept
        : red(inRed)
        , green(inGreen)
        , blue(inBlue)
        , alpha(inAlpha)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      32-bit RGBA値で初期化
     *  \param[in]  rgba    RGBA値
     */
    /* ------------------------------------------------------------------------- */
    constexpr Color(uint32_t rgba) noexcept
        : red(static_cast<float>(rgba >> 24u) / 255.0f)
        , green(static_cast<float>((rgba >> 16u) & 0xFFu) / 255.0f)
        , blue(static_cast<float>((rgba >> 8u) & 0xFFu) / 255.0f)
        , alpha(static_cast<float>(rgba & 0xFFu) / 255.0f)
    {}

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      xtermカラーで色を指定して初期化
     *  \param[in]  xcolor   xtermカラー
     */
    /* ------------------------------------------------------------------------- */
    constexpr Color(XColor xcolor) noexcept
        : Color(GetXColorCode(xcolor))
    {}
};

constexpr Color kColorRed = Color(1.0f, 0.0f, 0.0f, 1.0f);      //!< 赤
constexpr Color kColorGreen = Color(0.0f, 1.0f, 0.0f, 1.0f);    //!< 緑
constexpr Color kColorBlue = Color(0.0f, 0.0f, 1.0f, 1.0f);     //!< 青
constexpr Color kColorBlack = Color(0.0f, 0.0f, 0.0f, 1.0f);    //!< 黒
constexpr Color kColorWhite = Color(1.0f, 1.0f, 1.0f, 1.0f);    //!< 白

}    // namespace MGL

#endif    // INCGUARD_MGL_COLOR_H_1606630679

// vim: et ts=4 sw=4 sts=4
