// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_byte_stream.h
 *  \brief      バイトデータストリームクラス
 *  \date       Since: November 12, 2017. 2:49:43 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_BYTE_STREAM_H_1510422583
#define INCGUARD_MGL_BYTE_STREAM_H_1510422583

#include <cstdint>
#include <cstdlib>
#include <type_traits>

namespace MGL
{
//! バイトデータストリームクラス
class ByteStream
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr ByteStream() noexcept
        : _address(nullptr)
        , _size(0)
        , _offset(0)
        , _isOverflowed(false)
        , _isReadOnly(false)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  address 設定するバッファのアドレス
     *  \param[in]  size    バッファのサイズ
     */
    /* ------------------------------------------------------------------------- */
    constexpr ByteStream(void *address, size_t size) noexcept
        : _address(address)
        , _size(size)
        , _offset(0)
        , _isOverflowed(false)
        , _isReadOnly(false)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ（読み込み専用）
     *  \param[in]  address 設定するバッファのアドレス
     *  \param[in]  size    バッファのサイズ
     */
    /* ------------------------------------------------------------------------- */
    constexpr ByteStream(const void *address, size_t size) noexcept
        : _address(const_cast<void *>(address))
        , _size(size)
        , _offset(0)
        , _isOverflowed(false)
        , _isReadOnly(true)
    {
    }

    //! バッファクリア時の動作タイプ
    enum class ClearType : uint8_t
    {
        Zero,      //!< ゼロ初期化
        Random,    //!< ランダム初期化
    };

    void SetBuffer(void *address, size_t size) noexcept;
    void SetBuffer(const void *address, size_t size) noexcept;
    bool SetOffset(size_t offset) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      オフセットの取得
     *  \return     現在のオフセット
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr size_t GetOffset() const noexcept { return _offset; };

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      バッファサイズの取得
     *  \return     バッファサイズ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr size_t GetSize() const noexcept { return _size; };

    bool Clear(ClearType clearType) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      現在の先頭位置の取得
     *  \return     現在の先頭位置のアドレス
     *  \note
     *      バッファを読み込み専用に設定している場合はNULLが返る．
     *      その際は \ref GetCurrentConstPoint を使用する．
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr void *GetCurrentPoint() const noexcept
    {
        if (_isReadOnly)
        {
            return nullptr;
        }

        return &static_cast<uint8_t *>(_address)[_offset];
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      現在の先頭位置を取得（const版）
     *  \return     現在の先頭位置のアドレス
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const void *GetCurrentConstPoint() const noexcept
    {
        return &static_cast<const uint8_t *>(_address)[_offset];
    }

    bool Read(void *dest, size_t size) noexcept;
    bool Write(const void *source, size_t size) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      値の読み込み
     *  \param[in]  value   読み込む値
     *  \note
     *      valueの型のサイズ分だけストリームから読み込む．
     *      valueの型がsizeofで取得できる必要がある．
     */
    /* ------------------------------------------------------------------------- */
    template <typename T>
    constexpr bool Read(T &value) noexcept
    {
        if constexpr (std::is_same_v<bool, T>)
        {
            return ReadBool(value);
        }
        else
        {
            return Read(&value, sizeof(T));
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      型を指定して値を読み込み
     *  \tparam     T   読み込む型
     *  \note
     *      valueの型のサイズ分だけストリームから読み込む．
     *      valueの型がsizeofで取得できる必要がある．
     */
    /* ------------------------------------------------------------------------- */
    template <typename T>
    constexpr T Read() noexcept
    {
        T value = T();
        if constexpr (std::is_same_v<bool, T>)
        {
            ReadBool(&value);
        }
        else
        {
            Read(&value, sizeof(T));
        }
        return value;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      値の書き込み
     *  \param[in]  value   書き込む値
     *  \note
     *      valueの型のサイズ分だけストリームから読み込む．
     *      valueの型がsizeofで取得できる必要がある．
     */
    /* ------------------------------------------------------------------------- */
    template <typename T>
    constexpr bool Write(T value) noexcept
    {
        if constexpr (std::is_same_v<bool, T>)
        {
            return WriteBool(value);
        }
        else
        {
            return Write(&value, sizeof(T));
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      オーバーフローしているかを返す
     *  \retval     true    オーバーフローしている
     *  \retval     false   オーバーフローしていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsOverflowed() const noexcept
    {
        return _isOverflowed;
    }


    bool ReadBool(bool &flag) noexcept;
    bool ReadBool() noexcept;
    bool WriteBool(bool flag) noexcept;

    bool Fill(uint8_t value, size_t size) noexcept;
    bool Skip(size_t size) noexcept;

private:
    void *_address;
    size_t _size;
    size_t _offset;
    bool _isOverflowed;
    bool _isReadOnly;
};
}    // namespace MGL

#endif    // INCGUARD_MGL_BYTE_STREAM_H_1510422583

// vim: et ts=4 sw=4 sts=4
