// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_bit.h
 *  \brief      MGL ビットフラグ定義
 *  \date       Since: January 8, 2021. 10:07:57 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_BIT_H_1610068077
#define INCGUARD_MGL_BIT_H_1610068077

#include <cstdint>
#include <type_traits>

namespace MGL
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ビットフラグを生成
 *  \tparam     T   生成する型
 *  \param[in]  bit ビット番号
 *  \return     引数のビット番号をオンにしたT型の値
 */
/* ------------------------------------------------------------------------- */
template<typename T>
constexpr T MakeBit(uint8_t bit) noexcept
{
    return static_cast<T>(1u << bit);
}


/* ------------------------------------------------------------------------- */
/*!
 *  \brief      スコープを持つ列挙型に対応したビットフラグを扱うクラス
 *  \tparam     EnumType    スコープを持つ列挙型
 *  \tparam     FlagType    ビットフラグを保持するための値の型
 */
/* ------------------------------------------------------------------------- */
template<class EnumType, typename FlagType = uint32_t>
class EnumBitFlags
{
public:
    //! EnumTypeの基底の型
    using UnderlyingType = std::underlying_type_t<EnumType>;

    //! ビットフラグのサイズ
    static constexpr size_t kSize = sizeof(FlagType) * 8;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags() noexcept
        : _value(0)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  value   設定する値
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags(EnumType value) noexcept
        : _value(FlagType(1u << UnderlyingType(value)))
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  value   設定する値
     */
    /* ------------------------------------------------------------------------- */
    constexpr explicit EnumBitFlags(FlagType value) noexcept
        : _value(value)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フラグの値を取得
     *  \return     フラグの値の参照
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const FlagType &GetValue() const noexcept
    {
        return _value;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定した値を保持しているかを取得
     *  \param[in]  value   EnumTypeの型の値
     *  \retval     true    保持している
     *  \retval     false   保持していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool Has(EnumType value) const noexcept
    {
        return _value & FlagType(1u << UnderlyingType(value));
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      いずれかの値を保持しているかを取得
     *  \retval     true    保持している
     *  \retval     false   保持していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool HasAny() const noexcept
    {
        return _value != 0;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定した値をセットする
     *  \param[in]  value   EnumTypeの型の値
     *  \return     自身の参照
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags &Set(EnumType value) noexcept
    {
        _value |= FlagType(1u << UnderlyingType(value));
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定した値をクリアする
     *  \param[in]  value   EnumTypeの型の値
     *  \return     自身の参照
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags &Clear(EnumType value) noexcept
    {
        _value &= ~FlagType(1u << UnderlyingType(value));
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      全てのフラグをクリアする
     *  \return     自身の参照
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags &Clear() noexcept
    {
        _value = 0;
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定した値のフラグを反転させる
     *  \param[in]  value   EnumTypeの型の値
     *  \return     自身の参照
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags &Flip(EnumType value) noexcept
    {
        _value ^= FlagType(1 << UnderlyingType(value));
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      全ての値を反転させる
     *  \return     自身の参照
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags &Flip() noexcept
    {
        _value = ~_value;
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      bool型へのキャスト
     *  \retval     true    値が非ゼロである
     *  \retval     false   値がゼロである
     */
    /* ------------------------------------------------------------------------- */
    constexpr explicit operator bool() const noexcept
    {
        return _value != 0;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      否定演算子
     *  \retval     true    値がゼロである
     *  \retval     false   値が非ゼロである
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool operator!() const noexcept
    {
        return _value == 0;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      EnumType型との|=演算子
     *  \param[in]  rhs     EnumTypeの型の値
     *  \return     自身の参照
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags &operator|=(EnumType rhs) noexcept
    {
        _value |= FlagType(1 << UnderlyingType(rhs));
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      自身の型との|=演算子
     *  \param[in]  rhs     自身の型の値
     *  \return     自身の参照
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags &operator|=(const EnumBitFlags &rhs) noexcept
    {
        _value |= rhs._value;
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      EnumType型との&=演算子
     *  \param[in]  rhs     EnumTypeの型の値
     *  \return     自身の参照
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags &operator&=(EnumType rhs) noexcept
    {
        _value &= FlagType(1 << UnderlyingType(rhs));
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      自身の型との&=演算子
     *  \param[in]  rhs     自身の型の値
     *  \return     自身の参照
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags &operator&=(const EnumBitFlags &rhs) noexcept
    {
        _value &= rhs._value;
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      EnumType型との^=演算子
     *  \param[in]  rhs     EnumTypeの型の値
     *  \return     自身の参照
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags &operator^=(EnumType rhs) noexcept
    {
        _value ^= FlagType(1 << UnderlyingType(rhs));
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      自身の型との^=演算子
     *  \param[in]  rhs     自身の型の値
     *  \return     自身の参照
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags &operator^=(const EnumBitFlags &rhs) noexcept
    {
        _value ^= rhs._value;
        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      EnumType型との|演算子
     *  \param[in]  rhs     EnumTypeの型の値
     *  \return     演算結果
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags operator|(EnumType rhs) const noexcept
    {
        return EnumBitFlags(_value | FlagType(1u << UnderlyingType(rhs)));
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      自身の型との|演算子
     *  \param[in]  rhs     自身の型の値
     *  \return     演算結果
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags operator|(const EnumBitFlags &rhs) const noexcept
    {
        return EnumBitFlags(_value | rhs._value);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      EnumType型との&演算子
     *  \param[in]  rhs     EnumTypeの型の値
     *  \return     演算結果
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags operator&(EnumType rhs) const noexcept
    {
        return EnumBitFlags(_value & FlagType(1u << UnderlyingType(rhs)));
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      自身の型との&演算子
     *  \param[in]  rhs     自身の型の値
     *  \return     演算結果
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags operator&(const EnumBitFlags &rhs) const noexcept
    {
        return EnumBitFlags(_value & rhs._value);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      EnumType型との^演算子
     *  \param[in]  rhs     EnumTypeの型の値
     *  \return     演算結果
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags operator^(EnumType rhs) const noexcept
    {
        return EnumBitFlags(_value ^ FlagType(1u << UnderlyingType(rhs)));
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      自身の型との^演算子
     *  \param[in]  rhs     自身の型の値
     *  \return     演算結果
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags operator^(const EnumBitFlags &rhs) const noexcept
    {
        return EnumBitFlags(_value ^ rhs._value);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ~演算子(全てのビットの反転）
     *  \return     演算結果
     */
    /* ------------------------------------------------------------------------- */
    constexpr EnumBitFlags operator~() const noexcept
    {
        return EnumBitFlags(~_value);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      等価演算子
     *  \return     演算結果
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool operator==(const EnumBitFlags &rhs) const noexcept
    {
        return _value == rhs._value;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      不等価演算子
     *  \return     演算結果
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool operator!=(const EnumBitFlags &rhs) const noexcept
    {
        return _value != rhs._value;
    }

private:
    FlagType _value;
};
}    // namespace MGL
#endif    // INCGUARD_MGL_BIT_H_1610068077

// vim: et ts=4 sw=4 sts=4
