// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_alignment.h
 *  \brief      MGL 配置情報
 *  \date       Since: July 10, 2021. 16:01:08 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_ALIGNMENT_H_1625900468
#define INCGUARD_MGL_ALIGNMENT_H_1625900468

#include <mgl/math/mgl_vector2.h>
#include <mgl/math/mgl_rectangle.h>

namespace MGL
{
//! 配置情報
struct Alignment
{
    //! 水平方向の配置
    enum class Horizontal : uint8_t
    {
        Left,       //!< 左
        Center,     //!< 中央
        Right       //!< 右
    };

    //! 垂直方向の配置
    enum class Vertical : uint8_t
    {
        Top,        //!< 上
        Middle,     //!< 中央
        Bottom      //!< 下
    };

    Horizontal horizontal;  //!< 水平方向の配置情報
    Vertical vertical;      //!< 垂直方向の配置情報

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr Alignment() noexcept
        : horizontal(Horizontal::Left)
        , vertical(Vertical::Top)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  inHorizontal    水平方向の配置情報
     *  \param[in]  inVertical      垂直方向の配置情報
     */
    /* ------------------------------------------------------------------------- */
    constexpr Alignment(Horizontal inHorizontal, Vertical inVertical) noexcept
        : horizontal(inHorizontal)
        , vertical(inVertical)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  inVertical      垂直方向の配置情報
     *  \param[in]  inHorizontal    水平方向の配置情報
     */
    /* ------------------------------------------------------------------------- */
    constexpr Alignment(Vertical inVertical, Horizontal inHorizontal) noexcept
        : horizontal(inHorizontal)
        , vertical(inVertical)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      現在の配置情報を元に補正した座標を取得
     *  \param[in]  baseSize    基準となるサイズ
     *  \param[in]  position    座標
     *  \return     補正後の座標
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr MGL::Vector2 AdjustPosition(const MGL::Vector2 &baseSize, const MGL::Vector2 &position) const noexcept
    {
        MGL::Vector2 adjustedPosition(position);

        switch (horizontal)
        {
            case Horizontal::Left:
                break;

            case Horizontal::Center:
                adjustedPosition.x += baseSize.x * 0.5f;
                break;

            case Horizontal::Right:
                adjustedPosition.x += baseSize.x;
                break;
        }

        switch (vertical)
        {
            case Vertical::Top:
                break;

            case Vertical::Middle:
                adjustedPosition.y += baseSize.y * 0.5f;
                break;

            case Vertical::Bottom:
                adjustedPosition.y += baseSize.y;
                break;
        }

        return adjustedPosition;
    }
    
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      現在の配置情報を元に補正した矩形の原点を取得
     *  \param[in]  rectangle   矩形
     *  \return     補正後の矩形の原点
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr MGL::Vector2 AdjustRectangle(const MGL::Rectangle &rectangle) const noexcept
    {
        MGL::Vector2 adjustedPosition(rectangle.GetPosition());
        
        switch (horizontal)
        {
            case Horizontal::Left:
                break;
                
            case Horizontal::Center:
                adjustedPosition.x -= rectangle.width * 0.5f;
                break;
                
            case Horizontal::Right:
                adjustedPosition.x -= rectangle.width;
                break;
        }
        
        switch (vertical)
        {
            case Vertical::Top:
                break;
                
            case Vertical::Middle:
                adjustedPosition.y -= rectangle.height * 0.5f;
                break;
                
            case Vertical::Bottom:
                adjustedPosition.y -= rectangle.height;
                break;
        }
        
        return adjustedPosition;
    }
};

constexpr auto kAlignmentTopLeft         = Alignment(Alignment::Vertical::Top,     Alignment::Horizontal::Left);      //!< 左上
constexpr auto kAlignmentTopCenter       = Alignment(Alignment::Vertical::Top,     Alignment::Horizontal::Center);    //!< 中央上
constexpr auto kAlignmentTopRight        = Alignment(Alignment::Vertical::Top,     Alignment::Horizontal::Right);     //!< 右上
constexpr auto kAlignmentMiddleLeft      = Alignment(Alignment::Vertical::Middle,  Alignment::Horizontal::Left);      //!< 左中央
constexpr auto kAlignmentMiddleCenter    = Alignment(Alignment::Vertical::Middle,  Alignment::Horizontal::Center);    //!< 中央
constexpr auto kAlignmentMiddleRight     = Alignment(Alignment::Vertical::Middle,  Alignment::Horizontal::Right);     //!< 右中央
constexpr auto kAlignmentBottomLeft      = Alignment(Alignment::Vertical::Bottom,  Alignment::Horizontal::Left);      //!< 左下
constexpr auto kAlignmentBottomCenter    = Alignment(Alignment::Vertical::Bottom,  Alignment::Horizontal::Center);    //!< 中央下
constexpr auto kAlignmentBottomRight     = Alignment(Alignment::Vertical::Bottom,  Alignment::Horizontal::Right);     //!< 右下

}   // namespace MGL

#endif	// INCGUARD_MGL_ALIGNMENT_H_1625900468

// vim: et ts=4 sw=4 sts=4
