// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_array.h
 *  \brief      MGL 動的配列クラス
 *  \date       Since: April 12, 2025. 18:40:34 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_ARRAY_H_1744450834
#define INCGUARD_MGL_ARRAY_H_1744450834

#include <memory>
#include <type_traits>
#include <utility>

#include <mgl/common/mgl_range.h>
#include <mgl/memory/mgl_memory.h>
#include <mgl/memory/mgl_memory_utility.h>
#include <mgl/system/mgl_system_debug_macro.h>

namespace MGL
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      動的配列クラス
 *  \tparam     ValueType   配列が扱う型
 */
/* ------------------------------------------------------------------------- */
template <class ValueType, class IndexType = size_t>
class Array
{
public:
    // 配列の要素はデフォルト構築可能でなければならない
    static_assert(std::is_default_constructible_v<ValueType>, "Array element must be constructible by default.");

    // インデックスは整数型である必要がある
    static_assert(std::is_integral_v<IndexType>, "IndexType must be integral value.");

    // コピーは禁止
    Array(Array &) = delete;
    Array &operator=(Array &) = delete;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  arraySize   配列のサイズ
     *  \param[in]  clearMode   メモリ領域の初期化方法
     */
    /* ------------------------------------------------------------------------- */
    explicit Array(size_t arraySize = 0, Memory::ClearMode clearMode = Memory::ClearMode::Auto) noexcept
    {
        Allocate(arraySize, clearMode);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  arraySize       配列のサイズ
     *  \param[in]  invalidValue    無効値として設定する値
     *  \param[in]  clearMode       メモリ領域の初期化方法
     */
    /* ------------------------------------------------------------------------- */
    Array(size_t arraySize, const ValueType &invalidValue, Memory::ClearMode clearMode = Memory::ClearMode::Auto) noexcept
    {
        Allocate(arraySize, clearMode);
        SetInvalidValue(invalidValue);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  arraySize       配列のサイズ
     *  \param[in]  invalidValue    無効値として設定する値（右辺値参照）
     *  \param[in]  clearMode       メモリ領域の初期化方法
     */
    /* ------------------------------------------------------------------------- */
    Array(size_t arraySize, ValueType &&invalidValue, Memory::ClearMode clearMode = Memory::ClearMode::Auto) noexcept
    {
        Allocate(arraySize, clearMode);
        SetInvalidValue(std::move(invalidValue));
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ムーブコンストラクタ
     *  \param[in]  rhs     右辺値
     */
    /* ------------------------------------------------------------------------- */
    Array(Array &&rhs) noexcept
    {
        *this = std::move(rhs);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ムーブ代入
     *  \param[in]  rhs     右辺値
     */
    /* ------------------------------------------------------------------------- */
    Array &operator=(Array &&rhs) noexcept
    {
        Deallocate();

        // NOLINTBEGIN(cppcoreguidelines-prefer-member-initializer) : 上のDeallocate()が呼べなくなるので無理
        _arraySize = rhs._arraySize;
        _top = rhs._top;
        _tail = rhs._tail;
        // NOLINTEND(cppcoreguidelines-prefer-member-initializer)

        rhs._arraySize = 0;
        rhs._top = nullptr;
        rhs._tail = nullptr;

        return *this;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デストラクタ
     */
    /* ------------------------------------------------------------------------- */
    ~Array() noexcept
    {
        Deallocate();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      配列の生成
     *  \param[in]  arraySize       配列のサイズ
     *  \param[in]  clearMode       メモリ領域の初期化方法
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    bool New(size_t arraySize, Memory::ClearMode clearMode = Memory::ClearMode::Auto) noexcept
    {
        // 末尾（無効値）が存在している場合はそれを保つように生成
        if (_tail != nullptr)
        {
            auto invalidValue = std::move(*_tail);    // アロケート中に消えるので別のメモリ空間に移動しておく
            if (Allocate(arraySize, clearMode))
            {
                SetInvalidValue(std::move(invalidValue));
                return true;
            }
            return false;
        }
        // 末尾が存在していない場合は普通に生成
        else
        {
            // Note: 実はここには到達しない（コンストラクタがアロケートして無効値も生成されるため）
            return Allocate(arraySize, clearMode);
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      配列を同じサイズで構築し直す
     *  \param[in]  clearMode       メモリ領域の初期化方法
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    bool Renew(Memory::ClearMode clearMode = Memory::ClearMode::Auto) noexcept
    {
        return New(_arraySize, clearMode);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      全ての要素を指定値で書き込む
     *  \param[in]  value   書き込む値
     */
    /* ------------------------------------------------------------------------- */
    constexpr void Fill(const ValueType &value) noexcept
    {
        for (auto *e = begin(); e != end(); e++)
        {
            *e = value;
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      要素の取得
     *  \param[in]  index       取得する要素のインデックス
     *  \return     インデックスに対応した要素の参照。範囲外の場合は無効値。
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const ValueType &Get(IndexType index) const noexcept
    {
        return _top[GetIndex(index)];
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      添字による要素の取得
     *  \param[in]  index       取得する要素のインデックス
     *  \return     インデックスに対応した要素の参照。範囲外の場合は無効値。
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const ValueType &operator[](IndexType index) const noexcept
    {
        return Get(index);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      書き込み可能な要素へのポインタを取得
     *  \param[in]  index       取得する要素のインデックス
     *  \return     インデックスが範囲内であれば要素へのポインタ、範囲外の場合はnullptr。
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr ValueType *GetPtr(IndexType index) noexcept
    {
        if (auto i = GetIndex(index); i < _arraySize)
        {
            return std::addressof(_top[i]);
        }

        return nullptr;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      要素に値を設定
     *  \param[in]  index       設定する要素のインデックス
     *  \param[in]  value       設定する値
     *  \retval     true        成功
     *  \retval     false       失敗（範囲外）
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool Set(IndexType index, const ValueType &value) noexcept
    {
        if (auto *e = GetPtr(index); e != nullptr)
        {
            *e = value;
            return true;
        }

        return false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      値の書き込み
     *  \param[in]  index   書き込む値のインデックス
     *  \param[in]  body    値の書き込み関数
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    template <class Body>
    constexpr bool Write(IndexType index, Body body) noexcept
    {
        if (auto *e = GetPtr(index); e != nullptr)
        {
            if constexpr (std::is_invocable_r_v<void, decltype(body), ValueType &>)
            {
                body(*e);
                return true;
            }
            else
            {
                static_assert(std::false_type::value, "Write function is not matching arguments.");
                return false;
            }
        }

        return false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      要素に値をムーブ設定
     *  \param[in]  index       設定する要素のインデックス
     *  \param[in]  value       設定する値（右辺値参照）
     *  \retval     true        成功
     *  \retval     false       失敗（範囲外）
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool Set(IndexType index, ValueType &&value) noexcept
    {
        if (auto *e = GetPtr(index); e != nullptr)
        {
            *e = std::move(value);
            return true;
        }

        return false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      全ての要素へのアクセス
     *  \tparam     LoopBody    要素にアクセスするための関数オブジェクトの型
     *  \param[in]  body        要素にアクセスするための関数オブジェクト
     */
    /* ------------------------------------------------------------------------- */
    template <class LoopBody>
    constexpr void ForEach(LoopBody body) noexcept
    {
        for (size_t i = 0; i < _arraySize; i++)
        {
            if (!InvokeLoopBody(body, static_cast<IndexType>(i)))
            {
                return;
            }
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      全ての要素へ読み取り専用でアクセス
     *  \tparam     LoopBody    要素にアクセスするための関数オブジェクトの型
     *  \param[in]  body        要素にアクセスするための関数オブジェクト
     */
    /* ------------------------------------------------------------------------- */
    template <class LoopBody>
    constexpr void ForEach(LoopBody body) const noexcept
    {
        for (size_t i = 0; i < _arraySize; i++)
        {
            if (!InvokeLoopBody(body, static_cast<IndexType>(i)))
            {
                return;
            }
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      範囲を指定して要素にアクセス
     *  \tparam     LoopBody    要素にアクセスするための関数オブジェクトの型
     *  \param[in]  begin       開始インデックス
     *  \param[in]  end         終了インデックス
     *  \param[in]  body        要素にアクセスするための関数オブジェクト
     */
    /* ------------------------------------------------------------------------- */
    template <class LoopBody>
    constexpr void ForRange(IndexType begin, IndexType end, LoopBody body) noexcept
    {
        if (_arraySize == 0)
        {
            return;
        }

        Range<IndexType> range(
            std::clamp(begin, static_cast<IndexType>(0), static_cast<IndexType>(_arraySize) - 1),
            std::clamp(end, static_cast<IndexType>(0), static_cast<IndexType>(_arraySize) - 1));

        for (auto i = range.Begin(); range.Contains(i); i = range.Next(i))
        {
            if (!InvokeLoopBody(body, i))
            {
                return;
            }
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      範囲を指定して読み取り専用で要素にアクセス
     *  \tparam     LoopBody    要素にアクセスするための関数オブジェクトの型
     *  \param[in]  begin       開始インデックス
     *  \param[in]  end         終了インデックス
     *  \param[in]  body        要素にアクセスするための関数オブジェクト
     */
    /* ------------------------------------------------------------------------- */
    template <class LoopBody>
    constexpr void ForRange(IndexType begin, IndexType end, LoopBody body) const noexcept
    {
        if (_arraySize == 0)
        {
            return;
        }

        Range<IndexType> range(
            std::clamp(begin, 0, static_cast<IndexType>(_arraySize) - 1),
            std::clamp(end, 0, static_cast<IndexType>(_arraySize) - 1));

        for (auto i = range.Begin(); range.Contains(i); i = range.Next(i))
        {
            if (!InvokeLoopBody(body, i))
            {
                return;
            }
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定した要素が範囲内に含まれているかを取得
     *  \param[in]  element     チェックする要素の参照
     *  \retval     true        範囲内に含まれている要素である
     *  \retval     false       範囲内の要素ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool Contains(const ValueType &element) const noexcept
    {
        return Contains(std::addressof(element));
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定した要素が範囲内に含まれているかを取得
     *  \param[in]  element     チェックする要素のポインタ
     *  \retval     true        範囲内に含まれている要素である
     *  \retval     false       範囲内の要素ではない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool Contains(const ValueType *element) const noexcept
    {
        // Note:
        // ValueType型の実体をを内包するValueType型は作れないはずなので、
        // 各要素のアドレスと一致するかのチェックは不要なはず。

        auto topAddress = reinterpret_cast<uintptr_t>(_top);
        auto elemAddress = reinterpret_cast<uintptr_t>(element);
        auto tailAddress = reinterpret_cast<uintptr_t>(_tail);

        return (topAddress <= elemAddress) && (elemAddress < tailAddress);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      無効値の設定
     *  \param[in]  value   設定する値
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetInvalidValue(const ValueType &value) noexcept
    {
        if (_tail != nullptr)
        {
            *_tail = value;
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      無効値の設定
     *  \param[in]  value   設定する値（右辺値参照）
     */
    /* ------------------------------------------------------------------------- */
    constexpr void SetInvalidValue(ValueType &&value) noexcept
    {
        if (_tail != nullptr)
        {
            *_tail = std::move(value);
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      配列の要素数を取得
     *  \return     要素数
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr size_t GetSize() const noexcept
    {
        return _arraySize;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      先頭の要素を取得
     *  \return     先頭の要素
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr ValueType *begin() const noexcept
    {
        return _top;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      末尾の要素を取得
     *  \return     末尾の要素
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr ValueType *end() const noexcept
    {
        return _tail;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      先頭の要素をconstで取得
     *  \return     先頭の要素
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const ValueType *cbegin() const noexcept
    {
        return _top;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      末尾の要素をconstで取得
     *  \return     末尾の要素
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const ValueType *cend() const noexcept
    {
        return _tail;
    }

private:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      配列の生成（内部用）
     *  \param[in]  arraySize       配列のサイズ
     *  \param[in]  clearMode       メモリ領域の初期化方法
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    bool Allocate(size_t arraySize, Memory::ClearMode clearMode) noexcept
    {
        // 生成済みかつ同サイズであればアロケートを省略
        if ((_top != nullptr) && (_arraySize == arraySize))
        {
            // 全てのデストラクタを呼び出す
            std::destroy_n(_top, _arraySize + 1);

            // メモリ領域をクリア
            _top = Memory::Utility::InitializeArrayBuffer<ValueType>(_top, _arraySize + 1, clearMode);
        }
        // 未生成、またはサイズが異なる場合は一旦削除して際生成
        else
        {
            // 一旦削除
            Deallocate();

            // アロケータからメモリを確保
            const auto size = sizeof(ValueType) * (arraySize + 1);
            auto *ptr = Memory::Allocate(size);
            MGL_ASSERT(ptr, "Memory allocation failed.");
            if (ptr == nullptr)
            {
                return false;
            }

            // メモリ領域をクリア
            _top = Memory::Utility::InitializeArrayBuffer<ValueType>(ptr, arraySize + 1, clearMode);

            // 確保したメモリリソースをもとに各メンバを初期化
            _tail = std::addressof(_top[arraySize]);
            _arraySize = arraySize;
        }

        return true;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      配列の削除
     */
    /* ------------------------------------------------------------------------- */
    void Deallocate() noexcept
    {
        if (_top != nullptr)
        {
            std::destroy_n(_top, _arraySize + 1);
            Memory::Deallocate(_top);
            _top = _tail = nullptr;
            _arraySize = 0;
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      インデックスを取得
     *  \param[in]  index       変換元のインデックス
     *  \return     対応するインデックス。範囲外の場合は末尾のインデックス。
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr size_t GetIndex(IndexType index) const noexcept
    {
        // 符号付きの場合は負数もチェックする
        if constexpr (std::is_signed_v<IndexType>)
        {
            if ((index >= 0) && (static_cast<size_t>(index) < _arraySize))
            {
                return static_cast<size_t>(index);
            }
        }
        // 符号なしの場合はインデックス以下であるかどうかのみ
        else
        {
            if (static_cast<size_t>(index) < _arraySize)
            {
                return static_cast<size_t>(index);
            }
        }

        // 範囲外の場合は末尾のインデックスを返す
        return _arraySize;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ループ処理の本体の関数オブジェクトの呼び出し
     *  \tparam     LoopBody    要素にアクセスするための関数オブジェクトの型
     *  \param[in]  body        要素にアクセスするための関数オブジェクト
     *  \param[in]  index       インデックス
     *  \retval     true        継続
     *  \retval     false       中断
     */
    /* ------------------------------------------------------------------------- */
    template <class LoopBody>
    constexpr bool InvokeLoopBody(LoopBody body, IndexType index) noexcept
    {
        // インデックス付き、中断あり
        if constexpr (std::is_invocable_r_v<bool, decltype(body), IndexType, ValueType &>)
        {
            return body(index, _top[GetIndex(index)]);
        }
        // インデックス付き、中断なし
        else if constexpr (std::is_invocable_r_v<void, decltype(body), IndexType, ValueType &>)
        {
            body(index, _top[GetIndex(index)]);
            return true;
        }
        // インデックスなし、中断あり
        else if constexpr (std::is_invocable_r_v<bool, decltype(body), ValueType &>)
        {
            return body(_top[GetIndex(index)]);
        }
        // インデックスなし、中断なし
        else if constexpr (std::is_invocable_r_v<void, decltype(body), ValueType &>)
        {
            body(_top[GetIndex(index)]);
            return true;
        }
        // それ以外は不適格
        else
        {
            static_assert(std::false_type::value, "Loop body is not matching arguments.");
            return false;
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ループ処理の本体の関数オブジェクトを読み取り専用で呼び出し
     *  \tparam     LoopBody    要素にアクセスするための関数オブジェクトの型
     *  \param[in]  body        要素にアクセスするための関数オブジェクト
     *  \param[in]  index       インデックス
     *  \retval     true        継続
     *  \retval     false       中断
     */
    /* ------------------------------------------------------------------------- */
    template <class LoopBody>
    constexpr bool InvokeLoopBody(LoopBody body, IndexType index) const noexcept
    {
        // インデックス付き、中断あり
        if constexpr (std::is_invocable_r_v<bool, decltype(body), IndexType, ValueType &>)
        {
            return body(index, _top[GetIndex(index)]);
        }
        // インデックス付き、中断なし
        else if constexpr (std::is_invocable_r_v<void, decltype(body), IndexType, ValueType &>)
        {
            body(index, _top[GetIndex(index)]);
            return true;
        }
        // インデックスなし、中断あり
        else if constexpr (std::is_invocable_r_v<bool, decltype(body), ValueType &>)
        {
            return body(_top[GetIndex(index)]);
        }
        // インデックスなし、中断なし
        else if constexpr (std::is_invocable_r_v<void, decltype(body), ValueType &>)
        {
            body(_top[GetIndex(index)]);
            return true;
        }
        // それ以外は不適格
        else
        {
            static_assert(std::false_type::value, "Loop body is not matching arguments.");
            return false;
        }
    }

    size_t _arraySize{0};
    ValueType *_top{nullptr};
    ValueType *_tail{nullptr};
};
}    // namespace MGL

#endif    // INCGUARD_MGL_ARRAY_H_1744450834

// vim: et ts=4 sw=4 sts=4
