// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_wave_stream_voice.h
 *  \brief      MGL WAVEストリームボイス
 *  \date       Since: January 30, 2021. 10:15:20 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_AUDIO_WAVE_STREAM_VOICE_H_1611969320
#define INCGUARD_MGL_AUDIO_WAVE_STREAM_VOICE_H_1611969320

#include <mgl/audio/mgl_audio_stream_queue.h>
#include <mgl/audio/voice/mgl_audio_voice.h>
#include <mgl/audio/voice/mgl_audio_wave_loader.h>
#include <mgl/stl/mgl_stl_string.h>

#include <condition_variable>
#include <future>
#include <mutex>
#include <thread>

namespace MGL::Audio
{
//! MGL WAVEストリームボイス
class WaveStreamVoice : public Voice
{
public:
    WaveStreamVoice(VoiceKey key, const File::PathView &path, bool isLoop = false, uint32_t loopFrame = 0) noexcept;
    ~WaveStreamVoice() noexcept override;

    bool Load() noexcept override;
    bool GetSample(float &outDataL, float &outDataR) noexcept override;
    bool Start(uint32_t trackIndex, LoopType loopType) noexcept override;
    void Stop() noexcept override;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスの最大フレーム数を取得
     *  \param[in]  trackIndex  トラック番号
     *  \return     最大フレーム数
     */
    /* ------------------------------------------------------------------------- */
    uint32_t GetTotalFrame([[maybe_unused]] uint32_t trackIndex) const noexcept override
    {
        return _totalFrame;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したトラックのループ設定を取得
     *  \param[in]  trackIndex  トラック番号
     *  \retval     true        ループあり
     *  \retval     false       ループなし
     */
    /* ------------------------------------------------------------------------- */
    bool IsLoop([[maybe_unused]] uint32_t trackIndex) const noexcept override
    {
        return _isDefaultLoop;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したトラックのループフレームを取得
     *  \param[in]  trackIndex  トラック番号
     *  \return     ループフレーム
     */
    /* ------------------------------------------------------------------------- */
    uint32_t GetLoopFrame([[maybe_unused]] uint32_t trackIndex) const noexcept override
    {
        return _loopFrame;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスが持つトラックの数を取得
     *  \return     トラック数
     */
    /* ------------------------------------------------------------------------- */
    uint32_t GetTrackCount() const noexcept override
    {
        return 1;
    }

private:
    bool LoadWaveFile() noexcept;
    void ReadSample() noexcept;

    File::Path _path;
    uint32_t _totalFrame{0};
    uint32_t _loopFrame;
    WaveLoader _wave;
    StreamQueue _streamQueue;
    std::atomic<bool> _isBusy{false};
    std::atomic<bool> _isFinished{false};
    bool _isLoop{false};
    bool _isDefaultLoop;

    std::thread _readSampleThread;
    std::future<void> _loadFuture;

    std::mutex _mutex;
    std::condition_variable _condition;
};
}    // namespace MGL::Audio

#endif    // INCGUARD_MGL_AUDIO_WAVE_STREAM_VOICE_H_1611969320

// vim: et ts=4 sw=4 sts=4
