// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_wave_loader.h
 *  \brief      MGL WAVEローダー
 *  \date       Since: January 27, 2021. 16:09:26 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_AUDIO_WAVE_LOADER_H_1611731366
#define INCGUARD_MGL_AUDIO_WAVE_LOADER_H_1611731366

#include <mgl/file/mgl_file_throwing_handle.h>

namespace MGL::Audio
{
//! WAVEローダークラス
class WaveLoader
{
public:
    //! フォーマット
    struct Format
    {
        uint16_t type;             //!< データフォーマット（1でPCM）
        uint16_t channelCount;     //!< チャンネル数
        uint32_t samplesPerSec;    //!< サンプリングレート
        uint32_t bytesPerSec;      //!< 1秒あたりのバイト数
        uint16_t blockSize;        //!< 1サンプルのバイト数
        uint16_t bitsPerSample;    //!< 1サンプルあたりのビット数
    };
    static_assert(sizeof(Format) == 16);

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      WAVEファイルがオープンされているかを取得
     *  \retval     true    オープンされている
     *  \retval     false   オープンされていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsOpen() const noexcept
    {
        return _isOpen;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フォーマットを取得
     *  \return     フォーマット
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr const Format &GetFormat() const noexcept
    {
        return _format;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      読み込み位置を設定
     *  \param[in]  sampleFrame     設定するサンプルフレーム
     */
    /* ------------------------------------------------------------------------- */
    constexpr void Seek(uint32_t sampleFrame) noexcept
    {
        _currentDataOffset = static_cast<size_t>(sampleFrame) * _format.blockSize;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      読み込み位置が終端に達しているかを取得
     *  \retval     true    終端に達している
     *  \retval     false   終端に達していない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsFinished() const noexcept
    {
        return _currentDataOffset >= _dataSize;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      データサイズを取得
     *  \return     データサイズ（バイト数）
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr size_t GetDataSize() const noexcept
    {
        return _dataSize;
    }

    bool Open(const File::PathView &path) noexcept;
    void Close() noexcept;

    size_t GetSample(void *buffer, size_t bufferSize) noexcept;
    size_t GetSampleWithConvert(float *buffer, size_t bufferSize) noexcept;

private:
    bool _isOpen{false};
    Format _format{};
    File::ThrowingHandle _file;
    size_t _dataOffset{0};
    size_t _dataSize{0};
    size_t _currentDataOffset{0};
};
}    // namespace MGL::Audio

#endif    // INCGUARD_MGL_AUDIO_WAVE_LOADER_H_1611731366

// vim: et ts=4 sw=4 sts=4
