// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_voice.h
 *  \brief      MGL オーディオボイス
 *  \date       Since: January 16, 2021. 2:35:15 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_AUDIO_VOICE_H_1610732115
#define INCGUARD_MGL_AUDIO_VOICE_H_1610732115

#include <mgl/audio/mgl_audio_defs.h>
#include <mgl/mgl_environment.h>
#include <mgl/stl/mgl_stl_memory.h>

#include <atomic>

namespace MGL::Audio
{
//! MGL オーディオボイス
class Voice
{
public:
    //! ボイスの状態
    enum class Status : uint8_t
    {
        None,       //!< 状態なし（初期状態）
        Loading,    //!< 読み込み中
        Error,      //!< エラー発生
        Ready,      //!< 準備完了
        Remove      //!< 削除要求済み
    };

    //! ボイスのタイプ
    enum class Type : uint8_t
    {
        Static,    //!< スタティック: 読み込み後は内部状態が変化しない．同時再生可能
        Dynamic    //!< ダイナミック: 再生中に内部状態が変化する．同時再生不可能．ストリーム再生など
    };

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  key     ボイスキー
     *  \param[in]  type    ボイスのタイプ
     */
    /* ------------------------------------------------------------------------- */
    constexpr Voice(VoiceKey key, Type type) noexcept
        : _key(key)
        , _type(type)
        , _status(Status::None)
        , _volume(1.0f)
    {
    }

    virtual ~Voice() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスの読み込み処理
     *  \retval     true    成功
     *  \retval     false   失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool Load() noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      スタティックボイスのサンプルの取得
     *  \param[out] outDataL    左チャンネル出力の格納先
     *  \param[out] outDataR    右チャンネル出力の格納先
     *  \param[in]  trackIndex  トラック番号
     *  \param[in]  sampleFrame サンプルフレーム
     *  \retval     true        継続
     *  \retval     false       終了
     */
    /* ------------------------------------------------------------------------- */
    virtual bool GetSample([[maybe_unused]] float &outDataL, [[maybe_unused]] float &outDataR, [[maybe_unused]] uint32_t trackIndex, [[maybe_unused]] size_t sampleFrame) const noexcept
    {
        return false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ダイナミックボイスのサンプルの取得
     *  \param[out] outDataL    左チャンネル出力の格納先
     *  \param[out] outDataR    右チャンネル出力の格納先
     *  \retval     true        継続
     *  \retval     false       終了
     */
    /* ------------------------------------------------------------------------- */
    virtual bool GetSample([[maybe_unused]] float &outDataL, [[maybe_unused]] float &outDataR) noexcept
    {
        return false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスの開始処理
     *  \param[in]  trackIndex  トラック番号
     *  \param[in]  loopType    ループタイプ
     *  \retval     true        成功
     *  \retval     false       失敗
     */
    /* ------------------------------------------------------------------------- */
    virtual bool Start([[maybe_unused]] uint32_t trackIndex, [[maybe_unused]] LoopType loopType) noexcept
    {
        return true;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスの停止処理
     */
    /* ------------------------------------------------------------------------- */
    virtual void Stop() noexcept
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスの最大フレーム数を取得
     *  \param[in]  trackIndex  トラック番号
     *  \return     最大フレーム数
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual uint32_t GetTotalFrame(uint32_t trackIndex) const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したトラックのループ設定を取得
     *  \param[in]  trackIndex  トラック番号
     *  \retval     true        ループあり
     *  \retval     false       ループなし
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual bool IsLoop(uint32_t trackIndex) const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      指定したトラックのループフレームを取得
     *  \param[in]  trackIndex  トラック番号
     *  \return     ループフレーム
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual uint32_t GetLoopFrame(uint32_t trackIndex) const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスキーを取得
     *  \return     ボイスキー
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr VoiceKey GetKey() const noexcept
    {
        return _key;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      削除を要求
     */
    /* ------------------------------------------------------------------------- */
    void RemoveRequests() noexcept
    {
        switch (_status)
        {
            case Status::Ready:
                _status = Status::Remove;
                break;

            default:
                break;
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスのステータスを取得
     *  \return     ボイスのステータス
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] Status GetStatus() const noexcept
    {
        return _status;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスの種類を取得
     *  \return     ボイスの種類
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr Type GetType() const noexcept
    {
        return _type;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスが持つトラックの数を取得
     *  \return     トラック数
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] virtual uint32_t GetTrackCount() const noexcept = 0;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスのボリュームを設定
     *  \param[in]  volume  ボリューム
     */
    /* ------------------------------------------------------------------------- */
    void SetVolume(float volume) noexcept
    {
        _volume = volume;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスのボリュームを取得
     *  \return     volume  ボリューム
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float GetVolume() const noexcept
    {
        return _volume;
    }

protected:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスのステータスを設定
     *  \param[in]  status  設定するステータス
     */
    /* ------------------------------------------------------------------------- */
    void SetStatus(Status status) noexcept
    {
        _status = status;
    }

private:
    VoiceKey _key;
    Type _type;
    std::atomic<Status> _status;
    std::atomic<float> _volume;
};

//! 共有ボイスの型
using SharedVoice = std::shared_ptr<Voice>;

}    // namespace MGL::Audio
#endif    // INCGUARD_MGL_AUDIO_VOICE_H_1610732115

// vim: et ts=4 sw=4 sts=4
