// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_stream_queue.h
 *  \brief      MGL オーディオストリームキュー
 *  \date       Since: January 29, 2021. 17:20:41 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_AUDIO_STREAM_QUEUE_H_1611908441
#define INCGUARD_MGL_AUDIO_STREAM_QUEUE_H_1611908441

#include <condition_variable>
#include <mutex>

#include <mgl/stl/mgl_stl_memory.h>

namespace MGL::Audio
{
//! オーディオストリームキュークラス
class StreamQueue
{
public:
    //! キューバッファ情報
    struct QueueBuffer
    {
        STL::unique_ptr<float[]> data{nullptr};    //!< バッファアドレス
        uint32_t channelCount{0};                  //!< 1フレームのチャンネル数（1: モノラル，2: ステレオ）
        size_t dataSize{0};                        //!< データのサイズ
        QueueBuffer *next{nullptr};                //!< 次のバッファアドレス
        bool isActive{false};                      //!< 使用中フラグ
    };

    StreamQueue(size_t bufferSize, uint32_t bufferCount) noexcept;

    void Reset() noexcept;

    QueueBuffer *GetFreeBuffer() noexcept;
    bool AddBuffer(QueueBuffer *queueBuffer) noexcept;
    bool GetSample(float &frameDataL, float &frameDataR) noexcept;

    void Wait() noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      バッファサイズを取得
     *  \return     バッファサイズ
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr size_t GetBufferSize() const noexcept
    {
        return _bufferSize;
    }


private:
    STL::unique_ptr<QueueBuffer[]> _bufferArray;
    QueueBuffer *_top{nullptr};
    QueueBuffer *_tail{nullptr};

    size_t _bufferSize;
    uint32_t _bufferCount;
    uint32_t _activeBufferCount{0};
    uint32_t _currentFrameOffset{0};

    std::mutex _mutex;
    std::condition_variable _condition;
};
}    // namespace MGL::Audio

#endif    // INCGUARD_MGL_AUDIO_STREAM_QUEUE_H_1611908441

// vim: et ts=4 sw=4 sts=4
