// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_source_instance.h
 *  \brief      MGL オーディオソースインスタンス
 *  \date       Since: January 16, 2021. 3:32:38 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_AUDIO_SOURCE_INSTANCE_H_1610735558
#define INCGUARD_MGL_AUDIO_SOURCE_INSTANCE_H_1610735558

#include <atomic>

#include <mgl/audio/mgl_audio_fade.h>
#include <mgl/audio/voice/mgl_audio_voice.h>
#include <mgl/mgl_environment.h>

namespace MGL::Audio
{
//! MGL オーディオソースインスタンス
class SourceInstance
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  voice           ボイス
     *  \param[in]  isAutoRemove    再生終了後に自動で削除するかのフラグ
     */
    /* ------------------------------------------------------------------------- */
    SourceInstance(SharedVoice &voice, bool isAutoRemove) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デストラクタ
     */
    /* ------------------------------------------------------------------------- */
    ~SourceInstance() noexcept
    {
        _voice->Stop();
    }

    bool Play(uint32_t trackIndex, LoopType loopType, float volume) noexcept;
    bool Update(float &outDataL, float &outDataR) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      一時停止
     *  \param[in]  isEnabled   trueで一時停止，falseで解除
     */
    /* ------------------------------------------------------------------------- */
    void Pause(bool isEnabled) noexcept
    {
        _isPaused = isEnabled;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      一時停止の状態を取得
     *  \retval     true    一時停止中
     *  \retval     false   再生中
     */
    /* ------------------------------------------------------------------------- */
    bool IsPaused() const noexcept
    {
        return _isPaused;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      削除要求
     */
    /* ------------------------------------------------------------------------- */
    void Remove() noexcept
    {
        const std::scoped_lock lock(_mutex);

        _status = Status::Remove;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      音量の設定
     *  \param[in]  volume  音量
     *  \note       フェード期間中に呼び出した場合，フェードはキャンセルされる
     */
    /* ------------------------------------------------------------------------- */
    void SetVolume(float volume) noexcept
    {
        const std::scoped_lock lock(_mutex);

        _fade.Cancel();
        _volume = volume;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      変化時間を指定して音量を設定
     *  \param[in]  volume          音量
     *  \param[in]  fadeTimeSec     指定した音量に変化するまでの時間
     *  \param[in]  samplesPerSec   サンプリングレート
     *  \param[in]  isAutoStop      音量変化後に自動停止するかのフラグ
     *  \note       フェード期間中に呼び出した場合，指定したパラメータで上書きされる
     */
    /* ------------------------------------------------------------------------- */
    void SetVolume(float volume, float fadeTimeSec, float samplesPerSec, bool isAutoStop) noexcept
    {
        const std::scoped_lock lock(_mutex);

        _fade.Start(_volume, volume, fadeTimeSec, samplesPerSec, isAutoStop);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      音量を取得
     *  \return     現在の音量
     */
    /* ------------------------------------------------------------------------- */
    float GetVolume() const noexcept
    {
        return _volume;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      再生中かどうかを取得
     *  \retval     true    再生中
     *  \retval     false   再生中でない
     */
    /* ------------------------------------------------------------------------- */
    bool IsPlaying() const noexcept
    {
        return _status == Status::Play;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フェード中かどうかを取得
     *  \retval     true    フェード期間中
     *  \retval     false   フェード期間中でない
     */
    /* ------------------------------------------------------------------------- */
    constexpr bool IsFading() const noexcept
    {
        return _fade.IsActive();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      再生終了後に自動で削除するかの設定
     *  \param[in]  isEnabled   trueを指定すると自動削除が有効になる
     */
    /* ------------------------------------------------------------------------- */
    void SetAutoRemove(bool isEnabled) noexcept
    {
        _isAutoRemove = isEnabled;
    }

private:
    SharedVoice _voice;

    uint32_t _trackIndex{0};
    size_t _currentFrame{0};
    std::atomic<LoopType> _loopType{LoopType::ResourceDefault};

    std::atomic<bool> _isAutoRemove;
    std::atomic<bool> _isPaused{false};
    std::atomic<float> _volume{0.0f};

    //! ソースインスタンスのステータス
    enum class Status : uint8_t
    {
        Standby,    //!< スタンバイ
        Play,       //!< 再生中
        Remove      //!< 削除要求済み
    };
    std::atomic<Status> _status{Status::Standby};

    Fade _fade;

    std::mutex _mutex;
};

//! 共有ソースインスタンス
using SharedSourceInstance = std::shared_ptr<SourceInstance>;

//! 弱参照のソースインスタンス
using WeakSourceInstance = std::weak_ptr<SourceInstance>;

}    // namespace MGL::Audio

#endif    // INCGUARD_MGL_AUDIO_SOURCE_INSTANCE_H_1610735558

// vim: et ts=4 sw=4 sts=4
