// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_source.h
 *  \brief      MGL オーディオソース
 *  \date       Since: January 16, 2021. 6:03:09 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_AUDIO_SOURCE_H_1610744589
#define INCGUARD_MGL_AUDIO_SOURCE_H_1610744589

#include <mgl/audio/mgl_audio_player.h>
#include <mgl/audio/mgl_audio_source_instance.h>
#include <mgl/mgl_environment.h>

namespace MGL::Audio
{
//! オーディオソースクラス
class Source
{
public:
    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  sourceInstance  関連付けるオーディオソースインスタンス
     */
    /* ------------------------------------------------------------------------- */
    Source(const SharedSourceInstance &sourceInstance) noexcept
        : _sourceInstance(sourceInstance)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     */
    /* ------------------------------------------------------------------------- */
    constexpr Source() noexcept = default;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  voiceKey        ボイスキー
     *  \param[in]  isAutoRemove    再生終了時に自動で削除するかのフラグ
     */
    /* ------------------------------------------------------------------------- */
    Source(VoiceKey voiceKey, bool isAutoRemove = true) noexcept
        : Source()
    {
        Attach(voiceKey, isAutoRemove);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      デストラクタ
     */
    /* ------------------------------------------------------------------------- */
    ~Source() noexcept
    {
        Detach();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスとソースの関連付け
     *  \param[in]  voiceKey        ボイスキー
     *  \param[in]  isAutoRemove    再生終了時に自動で削除するかのフラグ
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    bool Attach(VoiceKey voiceKey, bool isAutoRemove) noexcept
    {
        Detach();
        _sourceInstance = Player::GetInstance().MakeSourceInstance(voiceKey, isAutoRemove);

        return true;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボイスとソースの関連付けを解除
     */
    /* ------------------------------------------------------------------------- */
    void Detach() noexcept
    {
        if (auto source = _sourceInstance.lock(); source != nullptr)
        {
            if (!source->IsPlaying() || source->IsPaused())
            {
                source->Remove();
            }
            else
            {
                source->SetAutoRemove(true);
            }

            _sourceInstance = WeakSourceInstance();
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ソースがボイスと関連付けられているかを取得
     *  \retval     true    関連付けられている
     *  \retval     false   関連付けられていない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsAttached() const noexcept
    {
        return !_sourceInstance.expired();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アタッチして再生
     *  \param[in]  voiceKey        ボイスキー
     *  \param[in]  isAutoRemove    再生終了時に自動で削除するかのフラグ
     *  \param[in]  trackIndex      トラック番号
     *  \param[in]  loopType        ループタイプ
     *  \param[in]  volume          再生音量
     *  \retval     true            成功
     *  \retval     false           失敗
     */
    /* ------------------------------------------------------------------------- */
    bool Play(VoiceKey voiceKey, bool isAutoRemove = true, uint32_t trackIndex = 0, LoopType loopType = LoopType::ResourceDefault, float volume = 1.0f) noexcept
    {
        if (!Attach(voiceKey, isAutoRemove))
        {
            return false;
        }

        return Play(trackIndex, loopType, volume);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      アタッチ済みのボイスを再生
     *  \param[in]  trackIndex  トラック番号
     *  \param[in]  loopType    ループタイプ
     *  \param[in]  volume      再生音量
     *  \retval     true        成功
     *  \retval     false       失敗
     */
    /* ------------------------------------------------------------------------- */
    bool Play(uint32_t trackIndex = 0, LoopType loopType = LoopType::ResourceDefault, float volume = 1.0f) noexcept
    {
        if (auto source = _sourceInstance.lock(); source != nullptr)
        {
            return source->Play(trackIndex, loopType, volume);
        }

        return false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      一時停止
     *  \param[in]  isEnabled   trueで一時停止，falseで解除
     */
    /* ------------------------------------------------------------------------- */
    void Pause(bool isEnabled = true) noexcept
    {
        if (auto source = _sourceInstance.lock(); source != nullptr)
        {
            source->Pause(isEnabled);
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      再開
     *  \note       Pause(false)と等価
     */
    /* ------------------------------------------------------------------------- */
    void Resume() noexcept
    {
        Pause(false);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      一時停止の状態を取得
     *  \retval     true    一時停止中
     *  \retval     false   再生中
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsPaused() const noexcept
    {
        if (const auto source = _sourceInstance.lock(); source != nullptr)
        {
            return source->IsPaused();
        }

        return false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      停止要求
     */
    /* ------------------------------------------------------------------------- */
    void Stop() noexcept
    {
        if (const auto source = _sourceInstance.lock(); source != nullptr)
        {
            source->Remove();
            _sourceInstance = WeakSourceInstance();
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      音量の設定
     *  \param[in]  volume  音量
     *  \note       フェード期間中は音量操作を行えず，この呼び出しは無視される
     */
    /* ------------------------------------------------------------------------- */
    void SetVolume(float volume) noexcept
    {
        if (const auto source = _sourceInstance.lock(); source != nullptr)
        {
            source->SetVolume(volume);
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      変化時間を指定して音量を設定
     *  \param[in]  volume          音量
     *  \param[in]  fadeTimeSec     指定した音量に変化するまでの時間
     *  \param[in]  isAutoStop      音量変化後に自動停止するかのフラグ
     *  \note       フェード期間中に呼び出した場合，指定したパラメータで上書きされる
     */
    /* ------------------------------------------------------------------------- */
    void SetVolume(float volume, float fadeTimeSec, bool isAutoStop = false) noexcept
    {
        if (const auto source = _sourceInstance.lock(); source != nullptr)
        {
            auto samplesPerSec = Player::GetInstance().GetOutputFormat().samplesPerSec;
            source->SetVolume(volume, fadeTimeSec, samplesPerSec, isAutoStop);
        }
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フェードアウト
     *  \param[in]  fadeTimeSec     フェード時間
     *  \param[in]  isAutoStop      音量変化後に自動停止するかのフラグ
     */
    /* ------------------------------------------------------------------------- */
    void Fadeout(float fadeTimeSec, bool isAutoStop = true) noexcept
    {
        SetVolume(0.0f, fadeTimeSec, isAutoStop);
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ボリュームの取得
     *  \return     現在のボリューム
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float GetVolume() const noexcept
    {
        if (const auto source = _sourceInstance.lock(); source != nullptr)
        {
            return source->GetVolume();
        }

        return 0.0f;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      再生中かどうかを取得
     *  \retval     true    再生中
     *  \retval     false   再生中でない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsPlaying() const noexcept
    {
        if (const auto source = _sourceInstance.lock(); source != nullptr)
        {
            return source->IsPlaying();
        }

        return false;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      フェード中かどうかを取得
     *  \retval     true    フェード期間中
     *  \retval     false   フェード期間中でない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] bool IsFading() const noexcept
    {
        if (const auto source = _sourceInstance.lock(); source != nullptr)
        {
            return source->IsFading();
        }

        return false;
    }

    // コピー禁止
    Source(const Source &) = delete;
    Source &operator=(const Source &) = delete;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ムーブコンストラクタ
     *  \param[in]  other   移動元ハンドル
     */
    /* ------------------------------------------------------------------------- */
    Source(Source &&other) noexcept
        : _sourceInstance(std::move(other._sourceInstance))
    {
        other._sourceInstance = WeakSourceInstance();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      ムーブ代入演算
     *  \param[in]  other   移動元ハンドル
     */
    /* ------------------------------------------------------------------------- */
    Source &operator=(Source &&other) noexcept
    {
        if (this != &other)
        {
            Detach();
            _sourceInstance = std::move(other._sourceInstance);
            other._sourceInstance = WeakSourceInstance();
        }

        return *this;
    }

private:
    WeakSourceInstance _sourceInstance;
};
}    // namespace MGL::Audio
#endif    // INCGUARD_MGL_AUDIO_SOURCE_H_1610744589

// vim: et ts=4 sw=4 sts=4
