// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_player.h
 *  \brief      MGL オーディオプレイヤー
 *  \date       Since: January 16, 2021. 3:53:23 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_AUDIO_PLAYER_H_1610736803
#define INCGUARD_MGL_AUDIO_PLAYER_H_1610736803

#include <mutex>

#include <mgl/audio/mgl_audio_defs.h>
#include <mgl/audio/mgl_audio_sampletype_convert.h>
#include <mgl/audio/mgl_audio_source_instance.h>
#include <mgl/audio/mgl_audio_voice_storage.h>
#include <mgl/audio/renderer/mgl_audio_renderer.h>
#include <mgl/common/mgl_singleton.h>
#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/stl/mgl_stl_memory.h>

namespace MGL
{
class InitializerDelegate;
}

namespace MGL::Audio
{
//! オーディオプレイヤークラス
class Player final : public SharedSingleton<Player>
{
public:
    static STL::unique_ptr<Player> &GetInstanceRef() noexcept;

    bool Initialize(STL::unique_ptr<Renderer> &renderer, InitializeMode initMode) noexcept;

    bool Load(const SharedVoice& voice) noexcept;
    bool Unload(VoiceKey key) noexcept;

    SharedSourceInstance MakeSourceInstance(VoiceKey key, bool isAutoRemove) noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      出力フォーマットの取得
     *  \return     出力フォーマット情報
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] const DataFormat &GetOutputFormat() const noexcept
    {
        return _renderer->GetOutputFormat();
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マスター音量の設定
     *  \param[in]  volume  設定する音量
     */
    /* ------------------------------------------------------------------------- */
    void SetMasterVolume(float volume) noexcept
    {
        _masterVolume = volume;
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      マスター音量の取得
     *  \return     現在のマスター音量
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] float GetMasterVolume() const noexcept
    {
        return _masterVolume;
    }

    void SetVoiceVolume(VoiceKey voiceKey, float volume) noexcept;
    [[nodiscard]] float GetVoiceVolume(VoiceKey voiceKey) noexcept;

private:
    static void Rendering(void **outData, size_t outDataCount, const DataFormat &dataFormat, size_t frameCount) noexcept;

    bool _isInitialized{false};
    VoiceStorage _voiceStorage;
    STL::list<SharedSourceInstance> _sources;
    std::mutex _mutex;

    std::atomic<float> _masterVolume{1.0f};

    SampleTypeConvert::WriteDataFunction _writeData{nullptr};

    STL::unique_ptr<Renderer> _renderer;
};
}    // namespace MGL::Audio

#endif    // INCGUARD_MGL_AUDIO_PLAYER_H_1610736803

// vim: et ts=4 sw=4 sts=4
