// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_audio_defs.h
 *  \brief      MGL オーディオ関連定義
 *  \date       Since: January 17, 2021. 5:28:47 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_AUDIO_DEFS_H_1610828927
#define INCGUARD_MGL_AUDIO_DEFS_H_1610828927

#include <cstddef>
#include <cstdint>

#include <mgl/hash/mgl_hash_fnv1a.h>

namespace MGL::Audio
{
//! ボイスキー
enum class VoiceKey : uint32_t {};

//! ボイスキーを生成する際のシード値
constexpr uint32_t kDefaultVoiceKeySeed = MGL::Hash::kFNV1aDefaultValue32;    // 32bitの値なら何でもいい．衝突する場合は変更を．

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ボイスキーを生成
 *  \param[in]  key     キーの文字列
 *  \param[in]  seed    シード値（省略時はkDefaultVoiceKeySeed）
 *  \return     キーから生成したハッシュ値
 */
/* ------------------------------------------------------------------------- */
constexpr VoiceKey MakeVoiceKey(const char *key, uint32_t seed = kDefaultVoiceKeySeed) noexcept
{
    return VoiceKey{MGL::Hash::FNV1a(key, seed)};
}

//! 初期化モード
enum class InitializeMode : uint8_t
{
    None,             //!< なし
    Sample44k1ch,     //!< 44KHz 1ch
    Sample44k2ch,     //!< 44KHz 2ch
    Sample22k1ch,     //!< 22KHz 1ch
    Sample22k2ch,     //!< 22KHz 2ch
    SystemDefault,    //!< システムのデフォルト
};

//! サンプルタイプ
enum class SampleType : uint8_t
{
    SignedInt,      //!< 符号付き整数
    UnsignedInt,    //!< 符号なし整数
    Float,          //!< 浮動小数点数
};

//! データフォーマット
struct DataFormat
{
    float samplesPerSec{0};                      //!< サンプリングレート
    SampleType sampleType{SampleType::Float};    //!< サンプルタイプ
    size_t bitsPerSample{0};                     //!< 1サンプルあたりのビット数
    size_t channelCount{0};                      //!< チャンネル数
    bool isInterleaved{false};                   //!< インターリーブかどうかのフラグ
};

//! ループタイプ
enum class LoopType : uint8_t
{
    Disable,           //!< 無効（ループしない）
    Enable,            //!< 有効（ループする）
    ResourceDefault    //!< リソースが持つ情報に委ねる
};

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      ループタイプをループフラグに変換
 *  \param[in]  loopType        ループタイプ
 *  \param[in]  defaultValue    LoopType::ResourceDefaultを指定した際の値
 *  \return     ループフラグ
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] constexpr bool ToLoopFlag(LoopType loopType, bool defaultValue = false) noexcept
{
    switch (loopType)
    {
        case LoopType::Disable:
            return false;

        case LoopType::Enable:
            return true;

        case LoopType::ResourceDefault:
            return defaultValue;
    }

    return defaultValue;
}
}    // namespace MGL::Audio

#endif    // INCGUARD_MGL_AUDIO_DEFS_H_1610828927

// vim: et ts=4 sw=4 sts=4
