// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_achievement_server.h
 *  \brief      MGL 実績サーバ
 *  \date       Since: September 8, 2021. 12:26:42 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_ACHIEVEMENT_SERVER_H_1631071602
#define INCGUARD_MGL_ACHIEVEMENT_SERVER_H_1631071602

#include <mgl/achievement/mgl_achievement_server_delegate.h>
#include <mgl/common/mgl_singleton.h>
#include <mgl/event/mgl_event.h>
#include <mgl/stl/mgl_stl_containers.h>
#include <mgl/stl/mgl_stl_memory.h>

namespace MGL::Achievement
{
//! MGL 実績サーバクラス
class Server final : public SharedSingleton<Server>
{
public:
    //! ステート
    enum class State : uint8_t
    {
        Initialize,    //!< 初期化中
        Idle,          //!< アイドル
        Fetch,         //!< 実績データを取得中
        Report,        //!< 実績データを送信中
    };

    static STL::unique_ptr<Server> &GetInstanceRef() noexcept;

    Server() noexcept;

    bool Initialize(STL::unique_ptr<ServerDelegate> &delegate) noexcept;

    bool SetValue(Identifier identifier, uint32_t value) noexcept;
    bool SetProgressRate(Identifier identifier, float progressRate) noexcept;
    bool SetAchieved(Identifier identifier, bool isAchieved) noexcept;

    [[nodiscard]] uint32_t GetValue(Identifier identifier) const noexcept;
    [[nodiscard]] float GetProgressRate(Identifier identifier) const noexcept;
    [[nodiscard]] bool IsAchieved(Identifier identifier) const noexcept;

    bool Clear() noexcept;

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      処理中かどうかを取得
     *  \retval     true    処理中
     *  \retval     false   処理中でない
     */
    /* ------------------------------------------------------------------------- */
    [[nodiscard]] constexpr bool IsProcessing() const noexcept
    {
        return (_state != State::Idle);
    }

private:
    static void OnPreFrameUpdate(void *callbackArg, void *notifyArg) noexcept;

    bool ApplyTemporaryData(const TemporaryData &data) noexcept;
    bool ApplyValue(Identifier identifier, uint32_t value) noexcept;
    bool ApplyProgressRate(Identifier identifier, float progressRate) noexcept;
    bool ApplyAchieved(Identifier identifier, bool isAchieved) noexcept;
    static void UpdateAchievementData(AchievementData *data) noexcept;

    STL::unique_ptr<ServerDelegate> _delegate;
    State _state{State::Initialize};
    STL::vector<TemporaryData> _temporaryDataArray;

    Event::Handle _eventPreFrameUpdate;
};
}    // namespace MGL::Achievement

#endif    // INCGUARD_MGL_ACHIEVEMENT_SERVER_H_1631071602

// vim: et ts=4 sw=4 sts=4
