// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_achievement_defs.h
 *  \brief      MGL 実績関連定義
 *  \date       Since: September 8, 2021. 12:54:15 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_ACHIEVEMENT_DEFS_H_1631073255
#define INCGUARD_MGL_ACHIEVEMENT_DEFS_H_1631073255

#include <mgl/mgl_environment.h>

#include <cstdint>

// MSVCではどこかでmaxという名前のマクロを定義してしまうらしい
#if defined(MGL_TARGET_WIN32) && defined(max)
#undef max
#endif

namespace MGL::Achievement
{
//! 実績用の識別子の型
using Identifier = uint32_t;

//! 無効な識別子
constexpr Identifier kInvalidIdentifier = 0xFFFFFFFF;

//! 値のタイプ
enum class ValueType : uint8_t
{
    Boolean,         //!< 論理値（解除 or 未解除）
    Integer,         //!< 整数値
    ProgressRate,    //!< 進行率（0.0f から 100.0f）
};

//! 実績データ
struct AchievementData
{
    ValueType type;              //!< 値のタイプ
    uint32_t max;                //!< タイプが整数値の場合の最大値
    uint32_t intValue{0};        //!< タイプが整数値の場合の現在値
    float progressRate{0.0f};    //!< 進行率
    bool isAchieved{false};      //!< 達成済みフラグ
    bool isUpdated{false};       //!< 値の更新フラグ

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      コンストラクタ
     *  \param[in]  type_   値のタイプ
     *  \param[in]  max_    値の最大値（タイプが整数値の場合のみ使用）
     */
    /* ------------------------------------------------------------------------- */
    constexpr AchievementData(ValueType type_, uint32_t max_ = 0) noexcept
        : type(type_)
        , max(max_)
    {
    }
};


//! 実績用のテンポラリデータ
struct TemporaryData
{
    Identifier identifier;    //!< 識別子
    ValueType type;           //!< 値のタイプ
    uint32_t intValue;        //!< 現在値（整数値の場合）
    bool isAchieved;          //!< 達成済みフラグ
    float progressRate;       //!< 進行率

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      論理値用コンストラクタ
     *  \param[in]  identifier_ 識別子
     *  \param[in]  isAchieved_ 達成済みフラグ
     */
    /* ------------------------------------------------------------------------- */
    constexpr TemporaryData(Identifier identifier_, bool isAchieved_) noexcept
        : identifier(identifier_)
        , type(ValueType::Boolean)
        , intValue(0)
        , isAchieved(isAchieved_)
        , progressRate(0.0f)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      整数値用コンストラクタ
     *  \param[in]  identifier_ 識別子
     *  \param[in]  value_      値
     */
    /* ------------------------------------------------------------------------- */
    constexpr TemporaryData(Identifier identifier_, uint32_t value_) noexcept
        : identifier(identifier_)
        , type(ValueType::Boolean)
        , intValue(value_)
        , isAchieved(false)
        , progressRate(0.0f)
    {
    }

    /* ------------------------------------------------------------------------- */
    /*!
     *  \brief      達成率用コンストラクタ
     *  \param[in]  identifier_     識別子
     *  \param[in]  progressRate_   達成率
     */
    /* ------------------------------------------------------------------------- */
    constexpr TemporaryData(Identifier identifier_, float progressRate_) noexcept
        : identifier(identifier_)
        , type(ValueType::Boolean)
        , intValue(0)
        , isAchieved(false)
        , progressRate(progressRate_)
    {
    }
};

}    // namespace MGL::Achievement

#endif    // INCGUARD_MGL_ACHIEVEMENT_DEFS_H_1631073255

// vim: et ts=4 sw=4 sts=4
