// SPDX-License-Identifier: Zlib
/* ------------------------------------------------------------------------- */
/*!
 *  \file       mgl_achievement.h
 *  \brief      MGL 実績
 *  \date       Since: September 13, 2021. 22:08:09 JST.
 *  \author     Acerola
 */
/* ------------------------------------------------------------------------- */

#ifndef INCGUARD_MGL_ACHIEVEMENT_H_1631538489
#define INCGUARD_MGL_ACHIEVEMENT_H_1631538489

#include <mgl/achievement/mgl_achievement_server.h>

namespace MGL::Achievement
{
/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実績サーバの生成
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
template <class DelegateClass, class... Args>
inline bool CreateServer(Args... args) noexcept
{
    if (Server::HasInstance())
    {
        return false;
    }

    STL::unique_ptr<ServerDelegate> delegate = STL::make_unique<DelegateClass>(args...);
    auto &server = Server::CreateInstance();

    return server.Initialize(delegate);
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サーバの有効状態を取得
 *  \retval     true    有効
 *  \retval     false   無効
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline bool IsAvailableServer() noexcept
{
    return Server::HasInstance();
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      値を設定
 *  \param[in]  identifier  実績の識別子
 *  \param[in]  value       設定する値
 *  \retval     true        成功
 *  \retval     false       失敗
 */
/* ------------------------------------------------------------------------- */
inline bool SetValue(Identifier identifier, uint32_t value) noexcept
{
    return IsAvailableServer() ? Server::GetInstance().SetValue(identifier, value) : false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      進捗率を設定
 *  \param[in]  identifier      実績の識別子
 *  \param[in]  progressRate    設定する進捗率
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
inline bool SetProgressRate(Identifier identifier, float progressRate) noexcept
{
    return IsAvailableServer() ? Server::GetInstance().SetProgressRate(identifier, progressRate) : false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      達成状態を設定
 *  \param[in]  identifier      実績の識別子
 *  \param[in]  isAchieved      設定する達成状態
 *  \retval     true            成功
 *  \retval     false           失敗
 */
/* ------------------------------------------------------------------------- */
inline bool SetAchieved(Identifier identifier, bool isAchieved) noexcept
{
    return IsAvailableServer() ? Server::GetInstance().SetAchieved(identifier, isAchieved) : false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実績の達成状況を整数値で取得
 *  \param[in]  identifier      実績の識別子
 *  \return     現在の達成状況
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline uint32_t GetValue(Identifier identifier) noexcept
{
    return IsAvailableServer() ? Server::GetInstance().GetValue(identifier) : 0;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実績の進捗率を取得
 *  \param[in]  identifier      実績の識別子
 *  \return     現在の進捗率
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline float GetProgressRate(Identifier identifier) noexcept
{
    return IsAvailableServer() ? Server::GetInstance().GetProgressRate(identifier) : 0.0f;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実績の達成状況を取得
 *  \param[in]  identifier      実績の識別子
 *  \retval     true            達成済み
 *  \retval     false           未達成
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline bool IsAchieved(Identifier identifier) noexcept
{
    return IsAvailableServer() ? Server::GetInstance().IsAchieved(identifier) : false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      実績のクリア
 *  \retval     true    成功
 *  \retval     false   失敗
 */
/* ------------------------------------------------------------------------- */
inline bool Clear() noexcept
{
    return IsAvailableServer() ? Server::GetInstance().Clear() : false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      処理中かどうかを取得
 *  \retval     true    処理中
 *  \retval     false   処理中でない
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline bool IsProcessing() noexcept
{
    return IsAvailableServer() ? Server::GetInstance().IsProcessing() : false;
}

/* ------------------------------------------------------------------------- */
/*!
 *  \brief      サーバが処理可能な状態かを取得
 *  \retval     true    処理中
 *  \retval     false   処理中でない
 */
/* ------------------------------------------------------------------------- */
[[nodiscard]] inline bool IsReady() noexcept
{
    return !IsProcessing();
}
}    // namespace MGL::Achievement

#endif    // INCGUARD_MGL_ACHIEVEMENT_H_1631538489

// vim: et ts=4 sw=4 sts=4
